<?php
/**
 * Plugin Name: Jenva Backup & Migration
 * Plugin URI: https://example.com/jenva-backup-migration
 * Description: Professional Backup & Migration plugin with batch backups, selective backup and domain migration
 * Version: 1.0.0
 * Author: Jenva Team
 * Author URI: https://example.com
 * Text Domain: jenva-backup-migration
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

/**
 * Prevent direct access / Direkten Zugriff verhindern
 */
if (!defined('ABSPATH')) {
    exit;
}

/**
 * CRITICAL: Set memory limits as early as possible (before WordPress load)
 * KRITISCH: Memory-Limits SO FRÜH WIE MÖGLICH setzen (vor WordPress-Load)
 * This helps with large sites with many plugins
 * Dies hilft bei großen Sites mit vielen Plugins
 */
if (isset($_GET['action']) && $_GET['action'] === 'jbm_download_backup') {
    @ini_set('memory_limit', '2048M'); // 2GB for large shops / 2GB für große Shops
    @set_time_limit(0);
    @ini_set('max_execution_time', 0);
    @ini_set('max_input_time', 0);
}

/**
 * Define plugin constants / Plugin-Konstanten definieren
 */
define('JBM_VERSION', '1.0.0');
define('JBM_PLUGIN_FILE', __FILE__);
define('JBM_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('JBM_PLUGIN_URL', plugin_dir_url(__FILE__));
define('JBM_BACKUP_DIR', WP_CONTENT_DIR . '/jenva-backup-migration/');
define('JBM_TEMP_DIR', JBM_BACKUP_DIR . 'temp/');
// Log-Verzeichnis (verwendet dasselbe wie Backup-Verzeichnis für Kompatibilität)
define('JBM_LOG_DIR', JBM_BACKUP_DIR);

/**
 * Autoloader for plugin classes / Autoloader für Plugin-Klassen
 */
spl_autoload_register(function ($class) {
    $prefix = 'JBM\\';
    $base_dir = JBM_PLUGIN_DIR . 'includes/';
    
    $len = strlen($prefix);
    if (strncmp($prefix, $class, $len) !== 0) {
        return;
    }
    
    $relative_class = substr($class, $len);
    $file = $base_dir . str_replace('\\', '/', $relative_class) . '.php';
    
    if (file_exists($file)) {
        require $file;
    }
});

/**
 * Main plugin class / Hauptklasse des Plugins
 */
final class Jenva_Backup_Migration {
    
    private static $instance = null;
    private $db_checked = false;
    
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->load_dependencies();
        $this->init_hooks();
    }
    
    private function init_hooks() {
        register_activation_hook(__FILE__, [$this, 'activate']);
        register_deactivation_hook(__FILE__, [$this, 'deactivate']);
        
        // Make plugin description translatable / Plugin-Beschreibung übersetzbar machen
        add_filter('all_plugins', [$this, 'translate_plugin_description']);
        
        // Register custom cron schedules (MUST be early!) / Custom Cron Schedules registrieren (MUSS FRÜH sein!)
        add_filter('cron_schedules', ['\JBM\Schedule', 'register_schedules']);
        
        // Initialize Schedule class (MUST be early for Cron hooks!) / Schedule-Klasse initialisieren (MUSS FRÜH sein für Cron-Hooks!)
        add_action('init', [$this, 'init_schedule'], 1); // Priority 1 = very early / Priorität 1 = sehr früh
        
        // Textdomain laden - nach set_user_locale() damit User-Sprache korrekt erkannt wird
        add_action('init', [$this, 'load_textdomain'], 5); // Priority 5 = nach set_user_locale()
        // Auch bei plugins_loaded als Fallback
        add_action('plugins_loaded', [$this, 'load_textdomain'], 10);
        add_action('plugins_loaded', [$this, 'check_database']);
        add_action('init', [$this, 'check_scheduled_backups_fallback'], 999); // Priorität 999 = spät
        
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_bar_styles']);
        add_action('admin_notices', [$this, 'admin_notices']);
        add_action('admin_bar_menu', [$this, 'add_admin_bar_backup_status'], 100);
        add_action('wp_ajax_jbm_create_backup', [$this, 'ajax_create_backup']);
        add_action('wp_ajax_jbm_restore_backup', [$this, 'ajax_restore_backup']);
        add_action('wp_ajax_jbm_delete_backup', [$this, 'ajax_delete_backup']);
        add_action('wp_ajax_jbm_download_backup', [$this, 'ajax_download_backup']);
        add_action('wp_ajax_jbm_save_schedule', [$this, 'ajax_save_schedule']);
        add_action('wp_ajax_jbm_get_backups', [$this, 'ajax_get_backups']);
        add_action('wp_ajax_jbm_fix_database', [$this, 'ajax_fix_database']);
        add_action('wp_ajax_jbm_get_live_logs', [$this, 'ajax_get_live_logs']);
        add_action('wp_ajax_jbm_toggle_protection', [$this, 'ajax_toggle_protection']);
        add_action('wp_ajax_jbm_cleanup_old_backups', [$this, 'ajax_cleanup_old_backups']);
        add_action('wp_ajax_jbm_test_schedule', [$this, 'ajax_test_schedule']);
        add_action('wp_ajax_jbm_diagnose_backup', [$this, 'ajax_diagnose_backup']);
        add_action('wp_ajax_jbm_diagnose_bricks', [$this, 'ajax_diagnose_bricks']);
        add_action('wp_ajax_jbm_upload_single_file', [$this, 'ajax_upload_single_file']);
        add_action('wp_ajax_jbm_upload_chunk', [$this, 'ajax_upload_chunk']);
        add_action('wp_ajax_jbm_clear_logs', [$this, 'ajax_clear_logs']);
        add_action('wp_ajax_jbm_get_backup_status', [$this, 'ajax_get_backup_status']);
        add_action('wp_ajax_jbm_hide_nginx_info', [$this, 'ajax_hide_nginx_info']);
    }
    
    /**
     * Translate plugin description in plugin list / Plugin-Beschreibung in Plugin-Liste übersetzen
     */
    public function translate_plugin_description($plugins) {
        $plugin_file = plugin_basename(__FILE__);
        
        if (isset($plugins[$plugin_file])) {
            $plugins[$plugin_file]['Description'] = __('Professional Backup & Migration plugin with batch backups, selective backup and domain migration', 'jenva-backup-migration');
        }
        
        return $plugins;
    }
    
    /**
     * Initialize Schedule class (important for Cron hooks) / Initialisiert die Schedule-Klasse (wichtig für Cron-Hooks)
     */
    public function init_schedule() {
        if (!class_exists('\JBM\Schedule')) {
            return;
        }
        
        // Initialize schedule - registers action hooks for Cron / Schedule initialisieren - registriert die Action-Hooks für Cron
        $schedule = new \JBM\Schedule();
        
        $logger = new \JBM\Logger();
        $logger->log("✅ Schedule-Klasse initialisiert - Cron-Hooks registriert");
    }
    
    /**
     * Fallback: Check on each page load if a backup is due / Fallback: Prüft bei jedem Seitenaufruf ob ein Backup fällig ist
     * Works even if WordPress Cron is disabled / Funktioniert auch wenn WordPress Cron deaktiviert ist
     */
    public function check_scheduled_backups_fallback() {
        // Only check every 5 minutes (performance) / Nur alle 5 Minuten prüfen (Performance)
        $last_check = get_transient('jbm_last_schedule_check');
        if ($last_check) {
            return;
        }
        set_transient('jbm_last_schedule_check', time(), 300); // 5 minutes / 5 Minuten
        
        // Prüfe beide Zeitpläne
        for ($i = 1; $i <= 2; $i++) {
            $schedule_data = get_option('jbm_schedule_' . $i);
            
            if (!$schedule_data || !$schedule_data['enabled']) {
                continue;
            }
            
            // Prüfe wann das letzte geplante Backup war
            $last_run = get_option('jbm_schedule_' . $i . '_last_run', 0);
            $now = time();
            
            // Zeitintervall basierend auf Frequenz
            $intervals = [
                'hourly' => 3600,
                'daily' => 86400,
                'weekly' => 604800,
                'monthly' => 2635200
            ];
            
            $interval = $intervals[$schedule_data['frequency']] ?? 86400;
            
            // Wenn genug Zeit vergangen ist, Backup ausführen
            if (($now - $last_run) >= $interval) {
                $logger = new \JBM\Logger();
                $logger->log("🔄 Fallback-Trigger: Zeitplan {$i} ist fällig");
                
                // Führe das geplante Backup aus (im Hintergrund)
                wp_schedule_single_event(time(), 'jbm_scheduled_backup_' . $i, [
                    $schedule_data['backup_type'],
                    $i,
                    $schedule_data
                ]);
                
                // Aktualisiere letzte Ausführung
                update_option('jbm_schedule_' . $i . '_last_run', $now);
                
                // Trigger WordPress Cron sofort
                spawn_cron();
            }
        }
    }
    
    private function load_dependencies() {
        // Legacy-Klassen (noch benötigt)
        require_once JBM_PLUGIN_DIR . 'includes/Database.php';
        require_once JBM_PLUGIN_DIR . 'includes/FileSystem.php';
        require_once JBM_PLUGIN_DIR . 'includes/Schedule.php';
        require_once JBM_PLUGIN_DIR . 'includes/Logger.php';
        
        // Neues 2.0 System - Autoloader laden
        require_once JBM_PLUGIN_DIR . 'includes/Autoloader.php';
    }
    
    /**
     * Plugin activation / Plugin-Aktivierung
     */
    public function activate() {
        $this->create_database_table();
        $this->create_directories();
        
        // Set default options (if not present) / Standard-Optionen setzen (falls nicht vorhanden)
        $existing_settings = get_option('jbm_settings');
        if (!$existing_settings) {
            add_option('jbm_settings', [
                'max_full_backups' => 10,
                'max_database_backups' => 30,
                'max_other_backups' => 20,
                'compression_level' => 6, // Fest: Mittel (Optimal)
                'backup_speed' => 'medium', // Fest: Mittel (Stabil)
                'email_notifications' => false,
                'email_address' => get_option('admin_email'),
                'show_diagnose_tool' => false,
                'show_cron_debug' => false,
            ]);
        } else {
            // Update bestehende Settings mit neuen Defaults
            $updated = false;
            if (!isset($existing_settings['max_full_backups'])) {
                $existing_settings['max_full_backups'] = 10;
                $updated = true;
            }
            if (!isset($existing_settings['max_database_backups'])) {
                $existing_settings['max_database_backups'] = 30;
                $updated = true;
            }
            if (!isset($existing_settings['max_other_backups'])) {
                $existing_settings['max_other_backups'] = 20;
                $updated = true;
            }
            if (!isset($existing_settings['show_diagnose_tool'])) {
                $existing_settings['show_diagnose_tool'] = false;
                $updated = true;
            }
            if (!isset($existing_settings['show_cron_debug'])) {
                $existing_settings['show_cron_debug'] = false;
                $updated = true;
            }
            // Migration: alte max_partial_backups entfernen
            if (isset($existing_settings['max_partial_backups'])) {
                unset($existing_settings['max_partial_backups']);
                $updated = true;
            }
            if ($updated) {
                update_option('jbm_settings', $existing_settings);
            }
        }
        
        // Markieren, dass Aktivierung durchgeführt wurde
        update_option('jbm_activated', true);
    }
    
    /**
     * Validiert und escaped einen Tabellennamen für sichere SQL-Queries
     * Sicherheit: Whitelist-Validierung für Plugin-Tabellennamen
     * 
     * @param string $table_name Der zu validierende Tabellenname
     * @return string|false Der validierte und escaped Tabellenname oder false
     */
    private function validate_table_name($table_name) {
        global $wpdb;
        
        // Nur unsere eigene Plugin-Tabelle erlauben
        $allowed_tables = [
            $wpdb->prefix . 'jbm_backups'
        ];
        
        if (!in_array($table_name, $allowed_tables, true)) {
            return false;
        }
        
        // Zusätzliche Validierung: Nur alphanumerische Zeichen und Unterstriche
        if (!preg_match('/^[a-zA-Z_][a-zA-Z0-9_]*$/', $table_name)) {
            return false;
        }
        
        // Backticks escapen
        return str_replace('`', '``', $table_name);
    }
    
    private function create_database_table() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'jbm_backups';
        $charset_collate = $wpdb->get_charset_collate();
        
        // Sicherheit: Tabellenname validieren
        $safe_table = $this->validate_table_name($table_name);
        if ($safe_table === false) {
            error_log('JBM: Ungültiger Tabellenname bei create_database_table');
            return false;
        }
        
        $sql = "CREATE TABLE IF NOT EXISTS `{$safe_table}` (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            backup_name varchar(255) NOT NULL,
            backup_type varchar(100) NOT NULL,
            backup_size bigint(20) DEFAULT 0,
            backup_date datetime DEFAULT CURRENT_TIMESTAMP,
            backup_path text NOT NULL,
            backup_status varchar(50) DEFAULT 'completed',
            backup_files longtext,
            protected tinyint(1) DEFAULT 0,
            PRIMARY KEY  (id),
            KEY backup_date (backup_date),
            KEY protected (protected)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        // Upgrade: Add column if not present / Upgrade: Spalte hinzufügen falls nicht vorhanden
        // Sicherheit: Prepared Statement für LIKE
        $columns = $wpdb->get_col(
            $wpdb->prepare("SHOW COLUMNS FROM `{$safe_table}` LIKE %s", 'protected')
        );
        if (empty($columns)) {
            $wpdb->query("ALTER TABLE `{$safe_table}` ADD protected tinyint(1) DEFAULT 0 AFTER backup_files");
        }
        
        // Check if table was created successfully / Prüfen, ob Tabelle erfolgreich erstellt wurde
        // Sicherheit: Prepared Statement für LIKE
        $table_exists = $wpdb->get_var(
            $wpdb->prepare("SHOW TABLES LIKE %s", $table_name)
        ) === $table_name;
        
        if ($table_exists) {
            delete_option('jbm_db_error');
        } else {
            update_option('jbm_db_error', 'Datenbanktabelle konnte nicht erstellt werden');
        }
        
        return $table_exists;
    }
    
    /**
     * Create plugin directories / Plugin-Verzeichnisse erstellen
     * Sicherheit: Umfassender Schutz für Apache UND Nginx
     */
    private function create_directories() {
        // Create backup directory / Backup-Verzeichnis erstellen
        if (!file_exists(JBM_BACKUP_DIR)) {
            wp_mkdir_p(JBM_BACKUP_DIR);
            wp_mkdir_p(JBM_TEMP_DIR);
        }
        
        // Sicherheitsdateien immer aktualisieren (für Updates)
        $this->create_directory_security_files();
    }
    
    /**
     * Erstellt Sicherheitsdateien für alle Plugin-Verzeichnisse
     * Sicherheit: Schützt vor direktem Zugriff auf Apache UND Nginx
     */
    private function create_directory_security_files() {
        // === BACKUP-VERZEICHNIS ===
        
        // .htaccess für Apache (2.2 und 2.4 kompatibel)
        $htaccess_content = "# Jenva Backup & Migration - Security\n";
        $htaccess_content .= "# Generiert am: " . date('Y-m-d H:i:s') . "\n\n";
        
        // Apache 2.4+ Syntax
        $htaccess_content .= "# Apache 2.4+\n";
        $htaccess_content .= "<IfModule mod_authz_core.c>\n";
        $htaccess_content .= "    # Standardmäßig alles blockieren\n";
        $htaccess_content .= "    Require all denied\n";
        $htaccess_content .= "    \n";
        $htaccess_content .= "    # Nur ZIP- und JBM-Dateien erlauben\n";
        $htaccess_content .= "    <FilesMatch \"\\.(zip|jbm)$\">\n";
        $htaccess_content .= "        Require all granted\n";
        $htaccess_content .= "    </FilesMatch>\n";
        $htaccess_content .= "</IfModule>\n\n";
        
        // Apache 2.2 Legacy Syntax
        $htaccess_content .= "# Apache 2.2 (Legacy)\n";
        $htaccess_content .= "<IfModule !mod_authz_core.c>\n";
        $htaccess_content .= "    Order deny,allow\n";
        $htaccess_content .= "    Deny from all\n";
        $htaccess_content .= "    <FilesMatch \"\\.(zip|jbm)$\">\n";
        $htaccess_content .= "        Allow from all\n";
        $htaccess_content .= "    </FilesMatch>\n";
        $htaccess_content .= "</IfModule>\n\n";
        
        // Zusätzliche Sicherheit
        $htaccess_content .= "# Verhindere Directory Listing\n";
        $htaccess_content .= "Options -Indexes\n\n";
        
        $htaccess_content .= "# Blockiere gefährliche Dateitypen explizit\n";
        $htaccess_content .= "<FilesMatch \"\\.(log|txt|json|sql|php|phtml|phar|sh|bash)$\">\n";
        $htaccess_content .= "    <IfModule mod_authz_core.c>\n";
        $htaccess_content .= "        Require all denied\n";
        $htaccess_content .= "    </IfModule>\n";
        $htaccess_content .= "    <IfModule !mod_authz_core.c>\n";
        $htaccess_content .= "        Order deny,allow\n";
        $htaccess_content .= "        Deny from all\n";
        $htaccess_content .= "    </IfModule>\n";
        $htaccess_content .= "</FilesMatch>\n";
        
        @file_put_contents(JBM_BACKUP_DIR . '.htaccess', $htaccess_content);
        
        // index.php für PHP-basierte Blockierung (funktioniert auch auf Nginx)
        $index_php = "<?php\n";
        $index_php .= "// Jenva Backup & Migration - Security\n";
        $index_php .= "// Direct access forbidden\n";
        $index_php .= "header('HTTP/1.0 403 Forbidden');\n";
        $index_php .= "exit('Forbidden');\n";
        @file_put_contents(JBM_BACKUP_DIR . 'index.php', $index_php);
        
        // index.html für Nginx (verhindert Directory Listing wenn autoindex on)
        $index_html = "<!DOCTYPE html>\n<html>\n<head>\n";
        $index_html .= "<meta charset=\"UTF-8\">\n";
        $index_html .= "<title>403 Forbidden</title>\n";
        $index_html .= "<style>body{font-family:Arial,sans-serif;text-align:center;padding:50px;}</style>\n";
        $index_html .= "</head>\n<body>\n";
        $index_html .= "<h1>403 Forbidden</h1>\n";
        $index_html .= "<p>You don't have permission to access this resource.</p>\n";
        $index_html .= "</body>\n</html>";
        @file_put_contents(JBM_BACKUP_DIR . 'index.html', $index_html);
        
        // === TEMP-VERZEICHNIS ===
        if (file_exists(JBM_TEMP_DIR)) {
            // Temp-Verzeichnis sollte KOMPLETT gesperrt sein
            $temp_htaccess = "# Jenva Backup & Migration - Temp Directory Security\n";
            $temp_htaccess .= "# KEIN Zugriff erlaubt!\n\n";
            $temp_htaccess .= "<IfModule mod_authz_core.c>\n";
            $temp_htaccess .= "    Require all denied\n";
            $temp_htaccess .= "</IfModule>\n";
            $temp_htaccess .= "<IfModule !mod_authz_core.c>\n";
            $temp_htaccess .= "    Order deny,allow\n";
            $temp_htaccess .= "    Deny from all\n";
            $temp_htaccess .= "</IfModule>\n";
            $temp_htaccess .= "Options -Indexes\n";
            @file_put_contents(JBM_TEMP_DIR . '.htaccess', $temp_htaccess);
            
            @file_put_contents(JBM_TEMP_DIR . 'index.php', $index_php);
            @file_put_contents(JBM_TEMP_DIR . 'index.html', $index_html);
        }
    }
    
    public function check_database() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'jbm_backups';
        
        // Sicherheit: Prepared Statement für LIKE-Clause
        $table_exists = $wpdb->get_var(
            $wpdb->prepare("SHOW TABLES LIKE %s", $table_name)
        ) === $table_name;
        
        if (!$table_exists) {
            update_option('jbm_db_error', 'Datenbanktabelle fehlt');
        } else {
            delete_option('jbm_db_error');
        }
    }
    
    public function admin_notices() {
        $db_error = get_option('jbm_db_error');
        
        if ($db_error && current_user_can('manage_options')) {
            ?>
            <div class="notice notice-error is-dismissible">
                <p>
                    <strong><?php _e('Jenva Backup & Migration:', 'jenva-backup-migration'); ?></strong> 
                    <?php _e('Die Datenbanktabelle fehlt! Backups können nicht gespeichert werden.', 'jenva-backup-migration'); ?>
                    <a href="#" id="bmp-fix-database" class="button button-primary" style="margin-left: 10px;">
                        <?php _e('Jetzt reparieren', 'jenva-backup-migration'); ?>
                    </a>
                </p>
            </div>
            <script>
                jQuery(document).ready(function($) {
                    $('#bmp-fix-database').on('click', function(e) {
                        e.preventDefault();
                        var button = $(this);
                        button.prop('disabled', true).text('<?php echo esc_js(__('Repariere...', 'jenva-backup-migration')); ?>');
                        
                        $.post(ajaxurl, {
                            action: 'jbm_fix_database',
                            nonce: '<?php echo wp_create_nonce('jbm_fix_database'); ?>'
                        }, function(response) {
                            if (response.success) {
                                button.closest('.notice').removeClass('notice-error').addClass('notice-success');
                                button.closest('.notice').find('p').html(
                                    '<strong><?php echo esc_js(__('Jenva Backup & Migration:', 'jenva-backup-migration')); ?></strong> ' + response.data.message
                                );
                                setTimeout(function() {
                                    location.reload();
                                }, 2000);
                            } else {
                                alert('<?php echo esc_js(__('Fehler:', 'jenva-backup-migration')); ?> ' + response.data.message);
                                button.prop('disabled', false).text('<?php echo esc_js(__('Jetzt reparieren', 'jenva-backup-migration')); ?>');
                            }
                        });
                    });
                });
            </script>
            <?php
        }
    }
    
    /**
     * Fügt Backup-Status zur Admin-Bar hinzu
     * Adds backup status to admin bar
     */
    public function add_admin_bar_backup_status($wp_admin_bar) {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Stelle sicher, dass Textdomain geladen ist / Ensure textdomain is loaded
        $this->load_textdomain();
        
        $backup_status = $this->get_backup_status();
        
        // Format: Jenva Backup | Status (gesamter Text farbig, nur Status animiert wenn nicht aktuell)
        if ($backup_status['is_current']) {
            $title = '<span style="color: #46b450; font-weight: 600;">Jenva Backup <span style="margin: 0 6px;">|</span> ✔ ' . __('Backup aktuell', 'jenva-backup-migration') . '</span>';
            $meta_class = 'jbm-backup-status-ok';
        } else {
            $title = '<span style="color: #dc3232; font-weight: 600;">Jenva Backup <span style="margin: 0 6px;">|</span> <span class="jbm-status-animated">⚠ ' . __('Backup nicht aktuell', 'jenva-backup-migration') . '</span></span>';
            $meta_class = 'jbm-backup-status-warning';
        }
        
        $wp_admin_bar->add_node([
            'id'    => 'jbm-backup-status',
            'title' => $title,
            'href'  => admin_url('admin.php?page=jenva-backup-migration'),
            'meta'  => [
                'class' => $meta_class,
                'title' => $backup_status['message'],
            ],
        ]);
    }
    
    /**
     * Ermittelt den Backup-Status basierend auf Website-Typ
     * Determines backup status based on website type
     */
    private function get_backup_status() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'jbm_backups';
        
        // Prüfe Website-Typ / Check website type
        $is_woocommerce = class_exists('WooCommerce');
        $post_count = wp_count_posts('post')->publish;
        $is_blog = $post_count > 20; // Mehr als 20 Artikel = Blog
        
        // Definiere erforderliche Backup-Intervalle basierend auf Website-Typ
        // Define required backup intervals based on website type
        if ($is_woocommerce) {
            // E-Commerce: DB stündlich, Full täglich
            $db_max_age = 3600; // 1 Stunde
            $full_max_age = 86400; // 24 Stunden
            $site_type = 'ecommerce';
        } elseif ($is_blog) {
            // Blog: DB täglich, Full wöchentlich
            $db_max_age = 86400; // 24 Stunden
            $full_max_age = 604800; // 7 Tage
            $site_type = 'blog';
        } else {
            // Kleine Website: DB täglich, Full wöchentlich
            $db_max_age = 86400; // 24 Stunden
            $full_max_age = 604800; // 7 Tage
            $site_type = 'small';
        }
        
        // Hole letztes DB-Backup / Get last DB backup
        $last_db_backup = $wpdb->get_row(
            "SELECT * FROM $table_name WHERE backup_type = 'database' ORDER BY backup_date DESC LIMIT 1"
        );
        
        // Hole letztes Full-Backup / Get last full backup
        $last_full_backup = $wpdb->get_row(
            "SELECT * FROM $table_name WHERE backup_type = 'full' ORDER BY backup_date DESC LIMIT 1"
        );
        
        $now = time();
        $is_current = true;
        $messages = [];
        
        // Prüfe DB-Backup / Check DB backup
        if ($last_db_backup) {
            $db_age = $now - strtotime($last_db_backup->backup_date);
            if ($db_age > $db_max_age) {
                $is_current = false;
                $messages[] = __('Datenbank-Backup überfällig', 'jenva-backup-migration');
            }
        } else {
            $is_current = false;
            $messages[] = __('Kein Datenbank-Backup vorhanden', 'jenva-backup-migration');
        }
        
        // Prüfe Full-Backup / Check full backup
        if ($last_full_backup) {
            $full_age = $now - strtotime($last_full_backup->backup_date);
            if ($full_age > $full_max_age) {
                $is_current = false;
                $messages[] = __('Vollständiges Backup überfällig', 'jenva-backup-migration');
            }
        } else {
            $is_current = false;
            $messages[] = __('Kein vollständiges Backup vorhanden', 'jenva-backup-migration');
        }
        
        // Erstelle Status-Nachricht / Create status message
        if ($is_current) {
            $message = __('Alle Backups sind aktuell', 'jenva-backup-migration');
        } else {
            $message = implode(' | ', $messages);
        }
        
        return [
            'is_current' => $is_current,
            'message' => $message,
            'site_type' => $site_type,
            'last_db_backup' => $last_db_backup,
            'last_full_backup' => $last_full_backup,
        ];
    }
    
    /**
     * AJAX-Handler für Backup-Status (für Live-Updates)
     * AJAX handler for backup status (for live updates)
     */
    public function ajax_get_backup_status() {
        check_ajax_referer('jbm_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Keine Berechtigung']);
        }
        
        $this->load_textdomain();
        $backup_status = $this->get_backup_status();
        
        // Format: Jenva Backup | Status
        if ($backup_status['is_current']) {
            $status_text = '✔ ' . __('Backup aktuell', 'jenva-backup-migration');
            $color = '#46b450';
            $class = 'jbm-backup-status-ok';
            $animated = false;
        } else {
            $status_text = '⚠ ' . __('Backup nicht aktuell', 'jenva-backup-migration');
            $color = '#dc3232';
            $class = 'jbm-backup-status-warning';
            $animated = true;
        }
        
        wp_send_json_success([
            'is_current' => $backup_status['is_current'],
            'status_text' => $status_text,
            'message' => $backup_status['message'],
            'color' => $color,
            'class' => $class,
            'animated' => $animated,
        ]);
    }
    
    public function ajax_fix_database() {
        check_ajax_referer('jbm_fix_database', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Keine Berechtigung']);
        }
        
        $success = $this->create_database_table();
        
        if ($success) {
            wp_send_json_success(['message' => 'Datenbanktabelle erfolgreich erstellt! Die Seite wird neu geladen...']);
        } else {
            wp_send_json_error(['message' => 'Datenbanktabelle konnte nicht erstellt werden. Bitte prüfen Sie die Datenbankrechte.']);
        }
    }
    
    public function deactivate() {
        // Geplante Backups entfernen
        wp_clear_scheduled_hook('jbm_scheduled_backup');
    }
    
    /**
     * Rate Limiting für kritische Operationen
     * Sicherheit: Verhindert DoS durch zu häufige Anfragen
     * 
     * @param string $action Name der Aktion (z.B. 'backup', 'restore', 'upload')
     * @param int $max_attempts Maximale Anzahl Versuche
     * @param int $window Zeitfenster in Sekunden
     * @return bool|array True wenn OK, array mit Fehlermeldung wenn Limit erreicht
     */
    private function check_rate_limit($action, $max_attempts = 5, $window = 60) {
        $user_id = get_current_user_id();
        $transient_key = 'jbm_rate_' . $action . '_' . $user_id;
        
        $data = get_transient($transient_key);
        
        if ($data === false) {
            // Erster Versuch
            set_transient($transient_key, ['count' => 1, 'first' => time()], $window);
            return true;
        }
        
        $count = $data['count'];
        $first = $data['first'];
        $elapsed = time() - $first;
        
        // Zeitfenster abgelaufen - Reset
        if ($elapsed >= $window) {
            set_transient($transient_key, ['count' => 1, 'first' => time()], $window);
            return true;
        }
        
        // Limit erreicht?
        if ($count >= $max_attempts) {
            $wait = $window - $elapsed;
            return [
                'limited' => true,
                'message' => sprintf(
                    __('Zu viele Anfragen. Bitte warten Sie %d Sekunden.', 'jenva-backup-migration'),
                    $wait
                ),
                'wait' => $wait,
            ];
        }
        
        // Counter erhöhen
        $data['count']++;
        set_transient($transient_key, $data, $window - $elapsed);
        
        return true;
    }
    
    /**
     * Validiert und bereinigt einen Upload-Dateinamen
     * Sicherheit: Verhindert Path Traversal und gefährliche Dateinamen
     * 
     * @param string $filename Der zu validierende Dateiname
     * @return string|false Der bereinigte Dateiname oder false bei ungültigem Namen
     */
    private function validate_upload_filename($filename) {
        // Null-Bytes und unsichtbare Zeichen entfernen
        $filename = preg_replace('/[\x00-\x1F\x7F]/', '', $filename);
        
        // Nur den Dateinamen extrahieren (entfernt Path Traversal wie ../ oder volle Pfade)
        $filename = basename($filename);
        
        // WordPress Sanitization
        $filename = sanitize_file_name($filename);
        
        // Prüfe ob noch etwas übrig ist
        if (empty($filename) || $filename === '.' || $filename === '..') {
            return false;
        }
        
        // Prüfe auf doppelte Erweiterungen (z.B. .php.zip -> wird zu .zip, aber wir wollen das erkennen)
        $dangerous_patterns = [
            '/\.php\./i',
            '/\.phtml\./i',
            '/\.phar\./i',
            '/\.htaccess/i',
            '/\.htpasswd/i',
            '/web\.config/i',
            '/\.ini$/i',
        ];
        
        foreach ($dangerous_patterns as $pattern) {
            if (preg_match($pattern, $filename)) {
                return false;
            }
        }
        
        // Maximale Länge
        if (strlen($filename) > 255) {
            $filename = substr($filename, 0, 250) . '.' . pathinfo($filename, PATHINFO_EXTENSION);
        }
        
        // Prüfe dass nur erlaubte Zeichen enthalten sind
        // Erlaubt: a-z, A-Z, 0-9, Bindestrich, Unterstrich, Punkt
        if (!preg_match('/^[a-zA-Z0-9._-]+$/', $filename)) {
            // Entferne nicht erlaubte Zeichen
            $filename = preg_replace('/[^a-zA-Z0-9._-]/', '_', $filename);
        }
        
        return $filename;
    }
    
    /**
     * Validiert Magic Bytes eines Chunks gegen die erwartete Dateiendung
     * Sicherheit: Verhindert das Hochladen von getarnten Dateien
     * 
     * @param string $file_path Pfad zur Chunk-Datei
     * @param string $expected_ext Erwartete Dateiendung (zip, gz, wpress)
     * @return bool True wenn Magic Bytes gültig sind
     */
    private function validate_chunk_magic_bytes($file_path, $expected_ext) {
        // Magic Bytes für verschiedene Formate
        $magic_signatures = [
            'zip' => [
                "\x50\x4B\x03\x04",  // PK.. (Standard ZIP)
                "\x50\x4B\x05\x06",  // PK.. (Leeres ZIP)
                "\x50\x4B\x07\x08",  // PK.. (Spanned ZIP)
            ],
            'jbm' => [
                // JBM-Dateien sind intern ZIP-Archive
                "\x50\x4B\x03\x04",  // PK.. (Standard ZIP)
                "\x50\x4B\x05\x06",  // PK.. (Leeres ZIP)
                "\x50\x4B\x07\x08",  // PK.. (Spanned ZIP)
            ],
            'gz' => [
                "\x1F\x8B",          // Gzip
            ],
            'wpress' => [
                // WPress-Dateien sind spezielle Archive, können mit verschiedenen Signaturen beginnen
                "\x50\x4B\x03\x04",  // Oft ZIP-basiert
                "SERVMASK",          // All-in-One WP Migration Header
            ],
        ];
        
        // Lese die ersten 16 Bytes
        $handle = @fopen($file_path, 'rb');
        if (!$handle) {
            return false;
        }
        
        $header = fread($handle, 16);
        fclose($handle);
        
        if (strlen($header) < 2) {
            return false;
        }
        
        // Prüfe gegen bekannte Signaturen
        $signatures = $magic_signatures[$expected_ext] ?? [];
        
        foreach ($signatures as $sig) {
            if (substr($header, 0, strlen($sig)) === $sig) {
                return true;
            }
        }
        
        // Für wpress-Dateien: Auch als Text-basiertes Format erlauben
        if ($expected_ext === 'wpress') {
            // Prüfe ob es möglicherweise ein gültiges WPRESS-Format ist
            // (manchmal beginnt es mit XML oder JSON)
            if (preg_match('/^[\x20-\x7E\r\n\t]/', $header)) {
                return true;
            }
        }
        
        // Unbekannte Signatur
        $logger = new \JBM\Logger();
        $logger->log('⚠️ Magic Bytes Validierung fehlgeschlagen für ' . $expected_ext . ': ' . bin2hex(substr($header, 0, 8)), 'warning');
        
        return false;
    }
    
    public function load_textdomain() {
        // Verwende get_user_locale() für Admin-Bereich, damit User-Sprache korrekt erkannt wird
        if (is_admin() && function_exists('get_user_locale')) {
            $locale = get_user_locale();
        } else {
            $locale = function_exists('determine_locale') ? determine_locale() : get_locale();
        }
        
        // Stelle sicher, dass die Locale korrekt ist
        if (empty($locale)) {
            $locale = 'en_US'; // Fallback
        }
        
        // Lade Textdomain mit korrekter Locale
        $locale = apply_filters('plugin_locale', $locale, 'jenva-backup-migration');
        
        // Prüfe ob Textdomain bereits mit dieser Locale geladen wurde
        static $loaded_locale = '';
        if ($loaded_locale === $locale) {
            return;
        }
        
        // Wenn bereits eine andere Locale geladen wurde, entlade sie
        if ($loaded_locale !== '') {
            unload_textdomain('jenva-backup-migration');
        }
        
        $loaded_locale = $locale;
        
        // .mo-Datei Pfad
        $lang_dir = plugin_dir_path(__FILE__) . 'languages/';
        $mofile_path = $lang_dir . 'jenva-backup-migration-' . $locale . '.mo';
        $pofile_path = $lang_dir . 'jenva-backup-migration-' . $locale . '.po';
        
        // Prüfe ob .po neuer als .mo ist und kompiliere neu
        $this->maybe_compile_mo($pofile_path, $mofile_path);
        
        // Prüfe ob .mo-Datei existiert, sonst Fallback auf en_US
        if (!file_exists($mofile_path) && $locale !== 'en_US') {
            $locale = 'en_US';
            $mofile_path = $lang_dir . 'jenva-backup-migration-en_US.mo';
            $pofile_path = $lang_dir . 'jenva-backup-migration-en_US.po';
            $this->maybe_compile_mo($pofile_path, $mofile_path);
        }
        
        // Lade die .mo-Datei direkt mit load_textdomain
        if (file_exists($mofile_path)) {
            load_textdomain('jenva-backup-migration', $mofile_path);
        } else {
            // Fallback auf load_plugin_textdomain
            load_plugin_textdomain('jenva-backup-migration', false, dirname(plugin_basename(__FILE__)) . '/languages');
        }
    }
    
    /**
     * Kompiliert .po zu .mo wenn .po neuer ist
     * Compiles .po to .mo if .po is newer
     */
    private function maybe_compile_mo($po_file, $mo_file) {
        if (!file_exists($po_file)) {
            return;
        }
        
        // Kompiliere wenn .mo nicht existiert oder .po neuer ist
        if (!file_exists($mo_file) || filemtime($po_file) > filemtime($mo_file)) {
            $this->compile_po_to_mo($po_file, $mo_file);
        }
    }
    
    /**
     * Kompiliert eine .po-Datei zu einer .mo-Datei
     * Compiles a .po file to a .mo file
     */
    private function compile_po_to_mo($po_file, $mo_file) {
        $translations = [];
        $content = file_get_contents($po_file);
        
        // Parse .po-Datei / Parse .po file
        // Multiline support
        preg_match_all('/msgid\s+"(.*)"\s*\nmsgstr\s+"(.*)"/Us', $content, $matches, PREG_SET_ORDER);
        
        foreach ($matches as $match) {
            $msgid = stripcslashes($match[1]);
            $msgstr = stripcslashes($match[2]);
            // Entferne führende/abschließende Newlines und verbinde mehrzeilige Strings
            $msgid = str_replace(["\"\n\"", "\"\r\n\""], '', $msgid);
            $msgstr = str_replace(["\"\n\"", "\"\r\n\""], '', $msgstr);
            if (!empty($msgid) && !empty($msgstr)) {
                $translations[$msgid] = $msgstr;
            }
        }
        
        if (empty($translations)) {
            return false;
        }
        
        // Erstelle .mo-Datei / Create .mo file
        $offsets = [];
        $ids = '';
        $strings = '';
        
        foreach ($translations as $id => $str) {
            $offsets[] = [strlen($ids), strlen($id), strlen($strings), strlen($str)];
            $ids .= $id . "\0";
            $strings .= $str . "\0";
        }
        
        $count = count($translations);
        $key_start = 28;
        $value_start = $key_start + $count * 8;
        $id_start = $value_start + $count * 8;
        $str_start = $id_start + strlen($ids);
        
        $mo = pack('Iiiiiii', 0x950412de, 0, $count, $key_start, $value_start, 0, 0);
        
        foreach ($offsets as $offset) {
            $mo .= pack('ii', $offset[1], $id_start + $offset[0]);
        }
        
        foreach ($offsets as $offset) {
            $mo .= pack('ii', $offset[3], $str_start + $offset[2]);
        }
        
        $mo .= $ids . $strings;
        
        return @file_put_contents($mo_file, $mo) !== false;
    }
    
    public function add_admin_menu() {
        add_menu_page(
            __('Jenva Backup & Migration', 'jenva-backup-migration'),
            __('Jenva Backup & Migration', 'jenva-backup-migration'),
            'manage_options',
            'jenva-backup-migration',
            [$this, 'render_main_page'],
            'dashicons-update',
            80
        );
        
        add_submenu_page(
            'jenva-backup-migration',
            __('Backups', 'jenva-backup-migration'),
            '🛟 ' . __('Backups', 'jenva-backup-migration'),
            'manage_options',
            'jenva-backup-migration',
            [$this, 'render_main_page']
        );
        
        add_submenu_page(
            'jenva-backup-migration',
            __('⏰ Zeitplan', 'jenva-backup-migration'),
            __('⏰ Zeitplan', 'jenva-backup-migration'),
            'manage_options',
            'backup-migration-schedule',
            [$this, 'render_schedule_page']
        );
        
        add_submenu_page(
            'jenva-backup-migration',
            __('⚙️ Einstellungen', 'jenva-backup-migration'),
            __('⚙️ Einstellungen', 'jenva-backup-migration'),
            'manage_options',
            'backup-migration-settings',
            [$this, 'render_settings_page']
        );
    }
    
    /**
     * Lädt Admin-Bar CSS auf allen Admin-Seiten
     * Loads admin bar CSS on all admin pages
     */
    public function enqueue_admin_bar_styles() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        wp_add_inline_style('admin-bar', '
            #wp-admin-bar-jbm-backup-status > .ab-item {
                display: flex !important;
                align-items: center !important;
            }
            #wp-admin-bar-jbm-backup-status:hover > .ab-item {
                background: rgba(0, 0, 0, 0.1) !important;
            }
            /* Nur der animierte Status-Teil pulsiert */
            .jbm-status-animated {
                animation: jbm-pulse 2s infinite;
            }
            @keyframes jbm-pulse {
                0%, 100% { opacity: 1; }
                50% { opacity: 0.5; }
            }
        ');
    }
    
    public function enqueue_admin_assets($hook) {
        if (strpos($hook, 'backup-migration') === false) {
            return;
        }
        
        // Stelle sicher, dass Textdomain geladen ist, bevor wir Übersetzungen verwenden
        $this->load_textdomain();
        
        wp_enqueue_style('jbm-admin-css', JBM_PLUGIN_URL . 'assets/css/admin.css', [], JBM_VERSION);
        wp_enqueue_script('jbm-admin-js', JBM_PLUGIN_URL . 'assets/js/admin.js', ['jquery'], JBM_VERSION, true);
        
        wp_localize_script('jbm-admin-js', 'jbmAdmin', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('jbm_nonce'),
            'strings' => [
                'creating' => __('Backup wird erstellt...', 'jenva-backup-migration'),
                'success' => __('Backup erfolgreich erstellt!', 'jenva-backup-migration'),
                'error' => __('Fehler beim Erstellen des Backups', 'jenva-backup-migration'),
                'confirm_delete' => __('Möchten Sie dieses Backup wirklich löschen?', 'jenva-backup-migration'),
                'confirm_restore' => __('Möchten Sie dieses Backup wirklich wiederherstellen? Die aktuelle Website wird überschrieben!', 'jenva-backup-migration'),
                'confirm_restore_title' => __('Backup wirklich wiederherstellen?', 'jenva-backup-migration'),
                'restoring' => __('Backup wird wiederhergestellt...', 'jenva-backup-migration'),
                'migrating' => __('Migration wird durchgeführt...', 'jenva-backup-migration'),
                'large_file_warning' => __('Dieses Backup ist sehr groß (%s).\n\nDer Download kann einige Minuten dauern.\n\nMöchten Sie fortfahren?', 'jenva-backup-migration'),
                'all_files_uploaded' => __('Alle %d Dateien erfolgreich hochgeladen!', 'jenva-backup-migration'),
                'file_uploading' => __('Datei %d von %d: %s', 'jenva-backup-migration'),
                'file_uploading_progress' => __('Datei %d von %d: %s (%s%%)', 'jenva-backup-migration'),
                'merging_files' => __('Führe %d Dateien zusammen...', 'jenva-backup-migration'),
                'merging_may_take_time' => __('Dies kann einige Minuten dauern...', 'jenva-backup-migration'),
                'merge_success' => __('Erfolgreich zusammengeführt!', 'jenva-backup-migration'),
                'merge_error' => __('Fehler beim Zusammenführen', 'jenva-backup-migration'),
                'critical_error' => __('Kritischer Fehler', 'jenva-backup-migration'),
                'upload_start' => __('Hochladen starten', 'jenva-backup-migration'),
                'selected_files' => __('Ausgewählte Dateien:', 'jenva-backup-migration'),
                'total' => __('Gesamt:', 'jenva-backup-migration'),
                'files_count' => __('Datei(en)', 'jenva-backup-migration'),
                'upload_running' => __('Upload läuft...', 'jenva-backup-migration'),
                'file_x_of_y' => __('Datei %d von %d', 'jenva-backup-migration'),
                'please_dont_close' => __('⏳ Bitte Seite nicht schließen...', 'jenva-backup-migration'),
                'restore_successful' => __('Wiederherstellung erfolgreich!', 'jenva-backup-migration'),
                'please_relogin' => __('ℹ️ Wichtig:', 'jenva-backup-migration'),
                'please_relogin_text' => __('Bitte melden Sie sich erneut an, um fortzufahren.', 'jenva-backup-migration'),
                'redirecting_login' => __('Weiterleitung zur Login-Seite in %d Sekunden...', 'jenva-backup-migration'),
                'url_migration_done' => __('🔄 URL-Migration durchgeführt:', 'jenva-backup-migration'),
                'from_url' => __('Von:', 'jenva-backup-migration'),
                'to_url' => __('Zu:', 'jenva-backup-migration'),
                'wpconfig_not_restored' => __('Die wp-config.php wird nicht wiederhergestellt. Die bestehende wp-config.php des Servers bleibt erhalten (beinhaltet korrekte Datenbank-Credentials).', 'jenva-backup-migration'),
                'hint' => __('Hinweis:', 'jenva-backup-migration'),
                'keep_existing_plugins_label' => __('Bestehende Plugins behalten', 'jenva-backup-migration'),
                'keep_existing_plugins_desc' => __('Plugins, die nicht im Backup enthalten sind, werden standardmäßig entfernt. Aktivieren Sie diese Option, um alle bestehenden Plugins zu behalten.', 'jenva-backup-migration'),
                'close_page_manual_login' => __('Sie können diese Seite jetzt schließen und sich manuell neu einloggen.', 'jenva-backup-migration'),
                'diagnose_test_running' => __('⏳ Test läuft... bitte warten...', 'jenva-backup-migration'),
                'diagnose_test_executing' => __('⏳ Führe Diagnose aus...', 'jenva-backup-migration'),
                'diagnose_ajax_error' => __('❌ AJAX-FEHLER:\n%s\nStatus: %s\n\nBitte Browser-Konsole (F12) prüfen!', 'jenva-backup-migration'),
                'diagnose_result' => __('📋 DIAGNOSE-ERGEBNIS:', 'jenva-backup-migration'),
                'diagnose_backup_create' => __('Backup::create() = ', 'jenva-backup-migration'),
                'diagnose_success' => __('SUCCESS', 'jenva-backup-migration'),
                'diagnose_failed' => __('FAILED', 'jenva-backup-migration'),
                'diagnose_backup_name' => __('Backup-Name: ', 'jenva-backup-migration'),
                'diagnose_zip_file' => __('ZIP-Datei ', 'jenva-backup-migration'),
                'diagnose_exists' => __('EXISTIERT', 'jenva-backup-migration'),
                'diagnose_not_exists' => __('EXISTIERT NICHT!', 'jenva-backup-migration'),
                'diagnose_path' => __('Pfad: ', 'jenva-backup-migration'),
                'diagnose_size' => __('Größe: ', 'jenva-backup-migration'),
                'diagnose_db_entry' => __('DB-Eintrag ', 'jenva-backup-migration'),
                'diagnose_present' => __('VORHANDEN (ID: ', 'jenva-backup-migration'),
                'diagnose_not_found' => __('NICHT GEFUNDEN!', 'jenva-backup-migration'),
                'diagnose_summary' => __('📊 ZUSAMMENFASSUNG:', 'jenva-backup-migration'),
                'diagnose_total_backups' => __('Gesamt Backups in DB: ', 'jenva-backup-migration'),
                'diagnose_dir_writable' => __('Backup-Dir beschreibbar: ', 'jenva-backup-migration'),
                'diagnose_yes' => __('Ja', 'jenva-backup-migration'),
                'diagnose_no' => __('NEIN!', 'jenva-backup-migration'),
                'diagnose_recent_backups' => __('📦 Letzte 5 Backups in DB:', 'jenva-backup-migration'),
                'diagnose_show_full_data' => __('🔍 Komplette Test-Daten anzeigen', 'jenva-backup-migration'),
                'diagnose_all_works' => __('🎉 ALLES FUNKTIONIERT! Die Diagnose-Box wird automatisch ausgeblendet.', 'jenva-backup-migration'),
                'diagnose_error' => __('❌ FEHLER: ', 'jenva-backup-migration'),
                'diagnose_ajax_error_title' => __('❌ AJAX-FEHLER:', 'jenva-backup-migration'),
                'diagnose_check_console' => __('Bitte Browser-Konsole (F12) prüfen!', 'jenva-backup-migration'),
                'size' => __('Größe', 'jenva-backup-migration'),
                'files' => __('Dateien', 'jenva-backup-migration'),
                'test_schedule_confirm' => __('Zeitplan %d jetzt manuell ausführen?\n\nDies erstellt sofort ein Backup und leitet Sie zur Backup-Liste weiter.', 'jenva-backup-migration'),
                'ajax_error' => __('❌ AJAX-Fehler!\n\n%s\n\nStatus: %s\n\nResponse: %s\n\nBitte Logs prüfen!', 'jenva-backup-migration'),
                'empty' => __('Leer', 'jenva-backup-migration'),
                'save_schedule_error' => __('Fehler beim Speichern des Zeitplans: %s\nStatus: %s\nBitte Browser-Konsole (F12) prüfen!', 'jenva-backup-migration'),
                'full' => __('Vollständige:', 'jenva-backup-migration'),
                'database_label' => __('Datenbank:', 'jenva-backup-migration'),
                'other' => __('Andere:', 'jenva-backup-migration'),
                'recommended_schedules' => __('💡 Empfohlene Zeitplan-Kombinationen', 'jenva-backup-migration'),
                'for_small_sites' => __('🏢 Für kleine Websites:', 'jenva-backup-migration'),
                'schedule_1_daily_db' => __('• Zeitplan 1: Täglich um 2:00 Uhr → Datenbank', 'jenva-backup-migration'),
                'schedule_2_weekly_full' => __('• Zeitplan 2: Wöchentlich Sonntag 3:00 Uhr → Vollständig', 'jenva-backup-migration'),
                'for_ecommerce' => __('🏪 Für E-Commerce/Shops:', 'jenva-backup-migration'),
                'schedule_1_hourly_db' => __('• Zeitplan 1: Stündlich → Datenbank (bei häufigen Bestellungen)', 'jenva-backup-migration'),
                'schedule_2_daily_full' => __('• Zeitplan 2: Täglich um 3:00 Uhr → Vollständig', 'jenva-backup-migration'),
                'for_blogs' => __('📰 Für Blogs/News-Seiten:', 'jenva-backup-migration'),
                'schedule_1_daily_db_alt' => __('• Zeitplan 1: Täglich um 1:00 Uhr → Datenbank', 'jenva-backup-migration'),
                'schedule_2_weekly_uploads' => __('• Zeitplan 2: Wöchentlich Montag 2:00 Uhr → Uploads', 'jenva-backup-migration'),
                'saved_schedules' => __('Gespeicherte Zeitpläne:', 'jenva-backup-migration'),
                'schedule' => __('Zeitplan', 'jenva-backup-migration'),
                'type' => __('Typ:', 'jenva-backup-migration'),
                'frequency' => __('Frequenz:', 'jenva-backup-migration'),
                'active' => __('✅ Aktiv', 'jenva-backup-migration'),
                'inactive' => __('❌ Inaktiv', 'jenva-backup-migration'),
                'next_execution' => __('Nächste Ausführung:', 'jenva-backup-migration'),
                'timestamp' => __('Timestamp:', 'jenva-backup-migration'),
                'in' => __('In:', 'jenva-backup-migration'),
                'args' => __('Args:', 'jenva-backup-migration'),
                'schedules_active' => __('✅ Zeitpläne sind aktiv und werden ausgeführt!', 'jenva-backup-migration'),
                'notes' => __('ℹ️ Hinweise:', 'jenva-backup-migration'),
                'cron_only_on_visit' => __('WordPress Cron läuft nur wenn Ihre Seite besucht wird', 'jenva-backup-migration'),
                'low_traffic_cron' => __('Bei wenig Traffic: Echten Server-Cron nutzen', 'jenva-backup-migration'),
                'no_events_resave' => __('Wenn keine Events sichtbar sind, Zeitplan neu speichern', 'jenva-backup-migration'),
                'manual_test' => __('🧪 Manuelle Zeitplan-Ausführung (Test):', 'jenva-backup-migration'),
                'manual_test_desc' => __('Führt das geplante Backup sofort aus, ohne auf den Zeitplan zu warten. Ideal zum Testen ob die Zeitplan-Logik funktioniert.', 'jenva-backup-migration'),
                'test_schedule_1' => __('🧪 Zeitplan 1 jetzt ausführen', 'jenva-backup-migration'),
                'test_schedule_2' => __('🧪 Zeitplan 2 jetzt ausführen', 'jenva-backup-migration'),
                'protected' => __('GESCHÜTZT', 'jenva-backup-migration'),
                'unlock' => __('Entsperren', 'jenva-backup-migration'),
                'protect' => __('Schützen', 'jenva-backup-migration'),
                'remove_protection' => __('Schutz aufheben', 'jenva-backup-migration'),
                'protect_from_deletion' => __('Vor Auto-Löschung schützen', 'jenva-backup-migration'),
                'protected_cannot_delete' => __('Geschützte Backups können nicht gelöscht werden', 'jenva-backup-migration'),
                'restore' => __('Wiederherstellen', 'jenva-backup-migration'),
                'download' => __('Herunterladen', 'jenva-backup-migration'),
                'delete' => __('Löschen', 'jenva-backup-migration'),
                'no_backups' => __('Keine Backups vorhanden', 'jenva-backup-migration'),
                'create_first_backup' => __('Erstellen Sie Ihr erstes Backup, um zu beginnen.', 'jenva-backup-migration'),
                'existing_backups' => __('Vorhandene Backups', 'jenva-backup-migration'),
                'error_deleting' => __('Fehler beim Löschen', 'jenva-backup-migration'),
                'delete_confirm_title' => __('Backup wirklich löschen?', 'jenva-backup-migration'),
                'delete_confirm_message' => __('Möchten Sie dieses Backup wirklich löschen?', 'jenva-backup-migration'),
                'delete_warning' => __('⚠️ WARNUNG: Gelöschte Backups können danach nicht wiederhergestellt werden!', 'jenva-backup-migration'),
                'delete_confirm_button' => __('Ja, löschen', 'jenva-backup-migration'),
                'delete_cancel_button' => __('Abbrechen', 'jenva-backup-migration'),
                'starting_download' => __('Starte Download...', 'jenva-backup-migration'),
                'restoring' => __('Wiederherstellen...', 'jenva-backup-migration'),
                'confirm_restore' => __('Möchten Sie dieses Backup wirklich wiederherstellen? Die aktuelle Website wird überschrieben.', 'jenva-backup-migration'),
                'confirm_clear_logs' => __('Möchten Sie wirklich alle Logs löschen?', 'jenva-backup-migration'),
                'clear_logs_title' => __('Logs wirklich löschen?', 'jenva-backup-migration'),
                'clear_logs_message' => __('Möchten Sie wirklich alle Logs löschen?', 'jenva-backup-migration'),
                'clear_logs_confirm_button' => __('Ja, löschen', 'jenva-backup-migration'),
                'clear_logs_cancel_button' => __('Abbrechen', 'jenva-backup-migration'),
                'logs_cleared' => __('Logs wurden erfolgreich gelöscht', 'jenva-backup-migration'),
                'error_clearing_logs' => __('Fehler beim Löschen der Logs', 'jenva-backup-migration'),
                'schedule_hints_title' => __('Hinweise:', 'jenva-backup-migration'),
                'schedule_cron_only_on_visit' => __('WordPress Cron läuft nur wenn Ihre Seite besucht wird', 'jenva-backup-migration'),
                'schedule_low_traffic_cron' => __('Bei wenig Traffic: Echten Server-Cron nutzen', 'jenva-backup-migration'),
                'schedule_no_events_resave' => __('Wenn keine Events sichtbar sind, Zeitplan neu speichern', 'jenva-backup-migration'),
                'schedule_manual_test_title' => __('🧪 Manuelle Zeitplan-Ausführung (Test):', 'jenva-backup-migration'),
                'schedule_manual_test_desc' => __('Führt das geplante Backup sofort aus, ohne auf den Zeitplan zu warten. Ideal zum Testen ob die Zeitplan-Logik funktioniert.', 'jenva-backup-migration'),
                'schedule_test_now_1' => __('🧪 Zeitplan 1 jetzt ausführen', 'jenva-backup-migration'),
                'schedule_test_now_2' => __('🧪 Zeitplan 2 jetzt ausführen', 'jenva-backup-migration'),
                'show_cron_debug' => __('Cron-Status & Debugging anzeigen', 'jenva-backup-migration'),
                'show_cron_debug_desc' => __('Zeigt den Cron-Status & Debugging-Bereich auf der Zeitplan-Seite an, um geplante Backups zu überwachen und zu testen.', 'jenva-backup-migration'),
                'backup_creating' => __('Dein Backup wird gerade erstellt...', 'jenva-backup-migration'),
                'backup_tip_regular' => __('Regelmäßige Backups sind der beste Schutz gegen Datenverlust!', 'jenva-backup-migration'),
                'backup_taking_time' => __('Es dauert noch ein bisschen...', 'jenva-backup-migration'),
                'backup_tip_hacked' => __('Wusstest du? 60% aller WordPress-Websites wurden bereits gehackt oder hatten Datenverlust.', 'jenva-backup-migration'),
                'backup_patience' => __('Bitte noch etwas Geduld...', 'jenva-backup-migration'),
                'backup_tip_save_work' => __('Ein gutes Backup kann Stunden oder Tage Arbeit retten!', 'jenva-backup-migration'),
                'backup_almost_done' => __('Noch ein wenig, gleich ist es geschafft...', 'jenva-backup-migration'),
                'backup_tip_321' => __('Wusstest du? Die 3-2-1 Regel: 3 Kopien, 2 Medien, 1 extern.', 'jenva-backup-migration'),
                'backup_fast_finished' => __('Fast fertig...', 'jenva-backup-migration'),
                'backup_tip_test' => __('Tipp: Teste deine Backups regelmäßig durch Wiederherstellung auf einer Test-Domain!', 'jenva-backup-migration'),
                'backup_last_step' => __('Letzter Schritt...', 'jenva-backup-migration'),
                'backup_tip_bankrupt' => __('Wusstest du? 93% der Unternehmen ohne Backup gehen nach einem Datenverlust bankrott.', 'jenva-backup-migration'),
                'backup_saving_data' => __('Deine Daten werden gesichert...', 'jenva-backup-migration'),
                'backup_tip_updates' => __('WordPress-Updates können manchmal schiefgehen - ein Backup ist Gold wert!', 'jenva-backup-migration'),
                'backup_creating_archive' => __('Backup-Archiv wird erstellt...', 'jenva-backup-migration'),
                'backup_tip_external' => __('Tipp: Lade wichtige Backups herunter und bewahre sie extern auf (Google Drive, Dropbox).', 'jenva-backup-migration'),
                'backup_running' => __('Sicherung läuft...', 'jenva-backup-migration'),
                'backup_tip_plugins' => __('Wusstest du? Plugin-Updates sind die häufigste Ursache für Website-Crashes.', 'jenva-backup-migration'),
                'backup_almost_there' => __('Gleich geschafft...', 'jenva-backup-migration'),
                'backup_tip_pro_tip' => __('Profi-Tipp: Erstelle vor jedem großen Update ein vollständiges Backup!', 'jenva-backup-migration'),
                'backup_please_wait' => __('💡 Bitte warten...', 'jenva-backup-migration'),
                'backup_success' => __('Backup erfolgreich erstellt!', 'jenva-backup-migration'),
                'backup_success_tip' => __('Deine Daten sind jetzt sicher gesichert. 🎉', 'jenva-backup-migration'),
                'backup_error' => __('Ein Fehler ist aufgetreten!', 'jenva-backup-migration'),
                'backup_error_tip' => __('Bitte prüfen Sie die Logs oder kontaktieren Sie den Support.', 'jenva-backup-migration'),
                'seconds_remaining' => __('Sekunden verbleibend', 'jenva-backup-migration'),
                'understood' => __('Verstanden', 'jenva-backup-migration'),
                'rate_limit_message' => __('Bitte warten Sie %d Sekunden vor dem nächsten Backup', 'jenva-backup-migration'),
                'rate_limit_reason' => __('Aus Sicherheitsgründen ist nur ein Backup alle 30 Sekunden erlaubt. Dies schützt Ihren Server vor Überlastung.', 'jenva-backup-migration'),
            ]
        ]);
    }
    
    // AJAX Handler
    public function ajax_create_backup() {
        try {
            check_ajax_referer('jbm_nonce', 'nonce');
            
            if (!current_user_can('manage_options')) {
                wp_send_json_error(['message' => __('Keine Berechtigung', 'jenva-backup-migration')]);
                return;
            }
            
            // Rate-Limiting: Max 1 Backup alle 30 Sekunden (Schutz vor DoS)
            $rate_limit_key = 'jbm_backup_rate_limit_' . get_current_user_id();
            $last_backup_time = get_transient($rate_limit_key);
            if ($last_backup_time && (time() - $last_backup_time) < 30) {
                $wait_time = 30 - (time() - $last_backup_time);
                wp_send_json_error([
                    'message' => sprintf(__('Bitte warten Sie %d Sekunden vor dem nächsten Backup', 'jenva-backup-migration'), $wait_time),
                    'rate_limited' => true,
                    'wait_time' => $wait_time,
                    'reason' => __('Aus Sicherheitsgründen ist nur ein Backup alle 30 Sekunden erlaubt. Dies schützt Ihren Server vor Überlastung.', 'jenva-backup-migration')
                ]);
                return;
            }
            set_transient($rate_limit_key, time(), 60);
            
            // Sicherstellen, dass Datenbanktabelle existiert
            $this->ensure_database_exists();
            
            $backup_type = isset($_POST['backup_type']) ? sanitize_text_field($_POST['backup_type']) : 'full';
            $backup_name = isset($_POST['backup_name']) ? sanitize_text_field($_POST['backup_name']) : '';
            
            // Neues 2.0 Backup-System verwenden
            $orchestrator = new \JenvaBackupMigration\Backup\BackupOrchestrator($backup_type, [
                'clean_database' => true,
                'exclude_cache' => true,
                'exclude_logs' => true,
                'exclude_other_backups' => true,
            ]);
            
            // Progress-Callback für Live-Updates
            $logger = new \JBM\Logger();
            $orchestrator->setProgressCallback(function($phase, $percent, $message) use ($logger) {
                $logger->log("📦 [{$phase}] {$percent}% - {$message}");
            });
            
            $result = $orchestrator->run();
            
            // Ergebnis in DB speichern (Kompatibilität mit bestehendem Interface)
            if ($result['success']) {
                global $wpdb;
                $table_name = $wpdb->prefix . 'jbm_backups';
                
                // Backup-Pfad: Die .jbm Datei (einzelne ZIP-ähnliche Datei)
                $backup_file_path = $result['jbm_path'] ?? $result['path'];
                
                // Backup-Name aus dem Dateinamen extrahieren
                $generated_name = pathinfo($backup_file_path, PATHINFO_FILENAME);
                
                // Dateigröße ermitteln
                $file_size = file_exists($backup_file_path) ? filesize($backup_file_path) : ($result['size'] ?? 0);
                
                
                $wpdb->insert(
                    $table_name,
                    [
                        'backup_name' => $generated_name,
                        'backup_type' => $backup_type,
                        'backup_size' => $file_size,
                        'backup_date' => current_time('mysql'),
                        'backup_path' => $backup_file_path,
                        'backup_status' => 'completed',
                        'protected' => 0,
                    ],
                    ['%s', '%s', '%d', '%s', '%s', '%s', '%d']
                );
                
                $result['backup_id'] = $wpdb->insert_id;
                $result['backup_name'] = $generated_name;
                $result['backup_path'] = $backup_file_path;
                $result['backup_size'] = $file_size;
                $result['message'] = __('Backup erfolgreich erstellt', 'jenva-backup-migration');
            }
            
            if ($result['success']) {
                wp_send_json_success($result);
            } else {
                wp_send_json_error($result);
            }
        } catch (\Exception $e) {
            $error_message = $e->getMessage();
            $error_file = $e->getFile();
            $error_line = $e->getLine();
            
            // Detailliertes Error-Logging
            if (class_exists('\JBM\Logger')) {
                $logger = new \JBM\Logger();
                $logger->log("❌ KRITISCHER FEHLER in ajax_create_backup", 'error');
                $logger->log("Fehlermeldung: {$error_message}", 'error');
                $logger->log("Datei: {$error_file}:{$error_line}", 'error');
                $logger->log("Stack Trace:\n" . $e->getTraceAsString(), 'error');
            }
            
            // Benutzerfreundliche Fehlermeldung
            $user_message = __('Fehler beim Erstellen des Backups', 'jenva-backup-migration');
            
            // Häufige Fehler mit Lösungsvorschlägen
            if (strpos($error_message, 'memory') !== false || strpos($error_message, 'Memory') !== false) {
                $user_message .= ': ' . __('Nicht genügend Speicher verfügbar', 'jenva-backup-migration');
                $user_message .= ' ' . __('Lösung: Erhöhen Sie memory_limit in php.ini', 'jenva-backup-migration');
            } elseif (strpos($error_message, 'timeout') !== false || strpos($error_message, 'Time') !== false) {
                $user_message .= ': ' . __('Zeitüberschreitung', 'jenva-backup-migration');
                $user_message .= ' ' . __('Lösung: Erhöhen Sie max_execution_time', 'jenva-backup-migration');
            } elseif (strpos($error_message, 'disk') !== false || strpos($error_message, 'space') !== false) {
                $user_message .= ': ' . __('Nicht genügend Speicherplatz', 'jenva-backup-migration');
            } else {
                $user_message .= ': ' . $error_message;
            }
            
            // Nur detaillierte Fehlerdetails bei WP_DEBUG
            $response = [
                'message' => $user_message,
            ];
            
            if (defined('WP_DEBUG') && WP_DEBUG) {
                $response['error_details'] = [
                    'file' => basename($error_file),
                    'line' => $error_line,
                    'trace' => $e->getTraceAsString()
                ];
            } else {
                $response['debug_hint'] = __('Weitere Details finden Sie im Aktivitätsprotokoll (Einstellungen → Aktivitätsprotokoll)', 'jenva-backup-migration');
            }
            
            wp_send_json_error($response);
        }
    }
    
    public function ajax_restore_backup() {
        check_ajax_referer('jbm_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Keine Berechtigung', 'jenva-backup-migration')]);
        }
        
        // Rate Limiting: Max 1 Restore alle 60 Sekunden (Schutz vor DoS)
        $rate_check = $this->check_rate_limit('restore', 1, 60);
        if (is_array($rate_check) && isset($rate_check['limited'])) {
            wp_send_json_error([
                'message' => $rate_check['message'],
                'rate_limited' => true,
                'wait_time' => $rate_check['wait'],
            ]);
            return;
        }
        
        $backup_id = isset($_POST['backup_id']) ? intval($_POST['backup_id']) : 0;
        $overwrite_wpconfig = isset($_POST['overwrite_wpconfig']) && $_POST['overwrite_wpconfig'] === 'true';
        $keep_existing_plugins = isset($_POST['keep_existing_plugins']) && $_POST['keep_existing_plugins'] === 'true';
        
        // Aktuelle User-Daten VORHER speichern (für Session-Wiederherstellung)
        $current_user = wp_get_current_user();
        $user_data = [
            'id' => $current_user->ID,
            'login' => $current_user->user_login,
            'email' => $current_user->user_email,
            'session_token' => wp_get_session_token(),
        ];
        
        // In Transient speichern (überlebt DB-Restore)
        set_transient('jbm_restore_user_' . $current_user->ID, $user_data, 3600);
        
        $logger = new \JBM\Logger();
        $logger->log("💾 User-Session gespeichert vor Restore: User ID " . $current_user->ID);
        if ($overwrite_wpconfig) {
            $logger->log("⚠️ wp-config.php wird überschrieben (vom Benutzer angefordert)");
        }
        if ($keep_existing_plugins) {
            $logger->log("ℹ️ Bestehende Plugins werden beibehalten (vom Benutzer angefordert)");
        } else {
            $logger->log("🗑️ Plugins, die nicht im Backup enthalten sind, werden entfernt");
        }
        
        try {
            // Backup-Pfad aus DB holen
            global $wpdb;
            $table_name = $wpdb->prefix . 'jbm_backups';
            $backup = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $backup_id));
            
            if (!$backup || !file_exists($backup->backup_path)) {
                wp_send_json_error(['message' => __('Backup nicht gefunden', 'jenva-backup-migration')]);
                return;
            }
            
            // Neues 2.0 Restore-System verwenden
            $wizard = new \JenvaBackupMigration\Restore\RestoreWizard($backup->backup_path, [
                'restore_database' => true,
                'restore_files' => true,
                'restore_plugins' => true,
                'migrate_urls' => true,
                'clear_caches' => true,
                'overwrite_wp_config' => $overwrite_wpconfig,
                'keep_existing_plugins' => $keep_existing_plugins, // false = entfernen (default), true = behalten
            ]);
            
            // Progress-Callback für Live-Updates
            $wizard->setProgressCallback(function($phase, $percent, $message) use ($logger) {
                $logger->log("🔄 [{$phase}] {$percent}% - {$message}");
            });
            
            $result = $wizard->run();
            
            if ($result['success']) {
                // Erfolgreich - Session sollte neu aufgebaut werden
                $result['require_relogin'] = true;
                $result['user_id'] = $current_user->ID;
                $result['message'] = __('Restore erfolgreich abgeschlossen', 'jenva-backup-migration');
                
                wp_send_json_success($result);
            } else {
                wp_send_json_error($result);
            }
        } catch (\Exception $e) {
            $logger->log("❌ Restore-Fehler: " . $e->getMessage(), 'error');
            wp_send_json_error([
                'message' => $e->getMessage(),
                'success' => false,
            ]);
        }
    }
    
    public function ajax_delete_backup() {
        check_ajax_referer('jbm_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Keine Berechtigung', 'jenva-backup-migration')]);
        }
        
        $backup_id = isset($_POST['backup_id']) ? intval($_POST['backup_id']) : 0;
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'jbm_backups';
        $backup = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $backup_id));
        
        if ($backup && file_exists($backup->backup_path)) {
            unlink($backup->backup_path);
        }
        
        $deleted = $wpdb->delete($table_name, ['id' => $backup_id], ['%d']);
        
        if ($deleted) {
            wp_send_json_success(['message' => __('Backup gelöscht', 'jenva-backup-migration')]);
        } else {
            wp_send_json_error(['message' => __('Fehler beim Löschen', 'jenva-backup-migration')]);
        }
    }
    
    public function ajax_download_backup() {
        // KRITISCH: Memory-Limit SO FRÜH WIE MÖGLICH setzen (vor WordPress-Load)
        // Dies wird zwar hier ausgeführt, aber wir setzen es trotzdem für maximale Kompatibilität
        @ini_set('memory_limit', '2048M'); // 2GB für große Shops
        @set_time_limit(0);
        @ini_set('max_execution_time', 0);
        @ini_set('max_input_time', 0);
        
        // Download erfolgt über GET, daher andere Nonce-Prüfung
        if (!isset($_GET['nonce']) || !wp_verify_nonce($_GET['nonce'], 'jbm_nonce')) {
            wp_die(__('Sicherheitsprüfung fehlgeschlagen', 'jenva-backup-migration'), '', ['response' => 403]);
        }
        
        if (!current_user_can('manage_options')) {
            wp_die(__('Keine Berechtigung', 'jenva-backup-migration'), '', ['response' => 403]);
        }
        
        $backup_id = isset($_GET['backup_id']) ? intval($_GET['backup_id']) : 0;
        
        if (!$backup_id) {
            wp_die(__('Ungültige Backup-ID', 'jenva-backup-migration'), '', ['response' => 400]);
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'jbm_backups';
        $backup = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $backup_id));
        
        if (!$backup) {
            wp_die(__('Backup nicht gefunden', 'jenva-backup-migration'), '', ['response' => 404]);
        }
        
        // Prüfe ob Datei existiert und lesbar ist
        if (!file_exists($backup->backup_path)) {
            $logger = new \JBM\Logger();
            $logger->log("❌ Download-Fehler: Datei nicht gefunden: " . $backup->backup_path, 'error');
            wp_die(__('Backup-Datei nicht gefunden', 'jenva-backup-migration'), '', ['response' => 404]);
        }
        
        if (!is_readable($backup->backup_path)) {
            $logger = new \JBM\Logger();
            $logger->log("❌ Download-Fehler: Datei nicht lesbar: " . $backup->backup_path, 'error');
            wp_die(__('Backup-Datei ist nicht lesbar', 'jenva-backup-migration'), '', ['response' => 500]);
        }
        
        // Dateigröße ermitteln (mit Fehlerbehandlung für sehr große Dateien)
        $file_size = @filesize($backup->backup_path);
        if ($file_size === false) {
            // Fallback: Versuche über stat() oder fseek()
            $file_size = 0;
            $file_handle = @fopen($backup->backup_path, 'rb');
            if ($file_handle) {
                @fseek($file_handle, 0, SEEK_END);
                $file_size = @ftell($file_handle);
                @fclose($file_handle);
            }
        }
        
        // Alle Output-Buffer leeren (mehrfach für verschachtelte Buffer)
        while (ob_get_level()) {
            ob_end_clean();
        }
        
        // Disable GZIP compression für Download (kann Probleme verursachen)
        if (function_exists('apache_setenv')) {
            @apache_setenv('no-gzip', 1);
        }
        @ini_set('zlib.output_compression', 'Off');
        
        // HTTP-Header setzen
        // Sicherheit: Dateiname sanitieren um Header-Injection zu verhindern
        $safe_filename = preg_replace('/[^a-zA-Z0-9_\-\.]/', '_', basename($backup->backup_path));
        header('Content-Type: application/zip', true);
        header('Content-Disposition: attachment; filename="' . $safe_filename . '"', true);
        if ($file_size > 0) {
            header('Content-Length: ' . $file_size, true);
        }
        header('Content-Transfer-Encoding: binary', true);
        header('Cache-Control: must-revalidate, post-check=0, pre-check=0, private', true);
        header('Pragma: public', true);
        header('Expires: 0', true);
        header('X-Accel-Buffering: no', true); // Nginx: Disable buffering
        
        // WICHTIG: Keine weiteren Ausgaben nach Headers
        // Datei in optimierten Chunks lesen (256KB statt 8KB für bessere Performance)
        $chunk_size = 262144; // 256KB - optimal für große Dateien
        $file_handle = @fopen($backup->backup_path, 'rb');
        
        if (!$file_handle) {
            $logger = new \JBM\Logger();
            $logger->log("❌ Download-Fehler: Konnte Datei nicht öffnen: " . $backup->backup_path, 'error');
            wp_die(__('Fehler beim Öffnen der Backup-Datei', 'jenva-backup-migration'), '', ['response' => 500]);
        }
        
        // Streaming-Download mit größeren Chunks
        $bytes_sent = 0;
        $start_time = microtime(true);
        
        try {
            while (!feof($file_handle)) {
                // Prüfe ob Verbindung noch aktiv ist
                if (connection_aborted()) {
                    break;
                }
                
                $chunk = @fread($file_handle, $chunk_size);
                if ($chunk === false) {
                    break;
                }
                
                echo $chunk;
                
                // Flush sofort (wichtig für große Dateien)
                if (ob_get_level()) {
                    ob_flush();
                }
                flush();
                
                $bytes_sent += strlen($chunk);
                
                // Memory freigeben (wichtig bei sehr großen Dateien)
                if ($bytes_sent % (10 * 1024 * 1024) == 0) { // Alle 10MB
                    if (function_exists('gc_collect_cycles')) {
                        @gc_collect_cycles();
                    }
                }
            }
            
            @fclose($file_handle);
            
            // Logging nur wenn Logger verfügbar (verhindert Fehler bei minimalem Load)
            if (class_exists('\JBM\Logger')) {
                $duration = round(microtime(true) - $start_time, 2);
                $logger = new \JBM\Logger();
                $logger->log("✅ Download erfolgreich: " . basename($backup->backup_path) . " (" . size_format($bytes_sent) . " in {$duration}s)");
            }
            
        } catch (\Exception $e) {
            @fclose($file_handle);
            if (class_exists('\JBM\Logger')) {
                $logger = new \JBM\Logger();
                $logger->log("❌ Download-Exception: " . $e->getMessage(), 'error');
            }
            // Kein wp_die hier, da Headers schon gesendet wurden
            exit;
        }
        
        exit;
    }
    
    public function ajax_upload_multiple_backups() {
        check_ajax_referer('jbm_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Keine Berechtigung', 'jenva-backup-migration')]);
        }
        
        if (empty($_FILES['backup_files'])) {
            wp_send_json_error(['message' => __('Keine Dateien hochgeladen', 'jenva-backup-migration')]);
        }
        
        $files = $_FILES['backup_files'];
        $file_count = count($files['name']);
        
        // Sicherheit: Nur .zip und .jbm Dateien erlauben (Dateiendung + MIME-Type + ZIP-Integrität)
        $allowed_extensions = ['zip', 'jbm'];
        $allowed_mimes = ['application/zip', 'application/x-zip-compressed', 'application/x-zip', 'application/octet-stream'];
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        
        for ($i = 0; $i < $file_count; $i++) {
            // Dateiendung prüfen
            $ext = strtolower(pathinfo($files['name'][$i], PATHINFO_EXTENSION));
            if (!in_array($ext, $allowed_extensions)) {
                finfo_close($finfo);
                wp_send_json_error(['message' => __('Nur ZIP- und JBM-Dateien sind erlaubt', 'jenva-backup-migration')]);
            }
            
            // MIME-Type prüfen
            $mime_type = finfo_file($finfo, $files['tmp_name'][$i]);
            if (!in_array($mime_type, $allowed_mimes)) {
                finfo_close($finfo);
                wp_send_json_error(['message' => __('Ungültiger Dateityp', 'jenva-backup-migration')]);
            }
            
            // ZIP-Integrität prüfen
            $zip = new \ZipArchive();
            if ($zip->open($files['tmp_name'][$i]) !== true) {
                finfo_close($finfo);
                wp_send_json_error(['message' => __('Ungültige oder beschädigte ZIP-Datei', 'jenva-backup-migration')]);
            }
            $zip->close();
        }
        finfo_close($finfo);
        
        $this->ensure_database_exists();
        
        $logger = new \JBM\Logger();
        $logger->log("Multi-File Upload gestartet: $file_count Datei(en)");
        
        try {
            $upload_dir = JBM_BACKUP_DIR;
            $uploaded_files = [];
            
            // Alle Dateien hochladen
            for ($i = 0; $i < $file_count; $i++) {
                if ($files['error'][$i] !== UPLOAD_ERR_OK) {
                    $logger->log('Upload-Fehler bei Datei ' . $files['name'][$i], 'error');
                    continue;
                }
                
                $filename = sanitize_file_name($files['name'][$i]);
                $tmp_name = $files['tmp_name'][$i];
                
                // SICHERHEIT: Nur ZIP-Dateien erlauben
                $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
                if ($ext !== 'zip') {
                    $logger->log('Sicherheit: Nicht-ZIP Datei abgelehnt: ' . $filename, 'warning');
                    continue;
                }
                
                // SICHERHEIT: MIME-Type validieren
                $finfo = finfo_open(FILEINFO_MIME_TYPE);
                $mime_type = finfo_file($finfo, $tmp_name);
                finfo_close($finfo);
                $allowed_mimes = ['application/zip', 'application/x-zip-compressed', 'application/x-zip', 'application/octet-stream'];
                if (!in_array($mime_type, $allowed_mimes)) {
                    $logger->log('Sicherheit: Ungültiger MIME-Type abgelehnt: ' . $mime_type, 'warning');
                    continue;
                }
                
                // SICHERHEIT: ZIP-Integrität prüfen
                $zip = new \ZipArchive();
                if ($zip->open($tmp_name) !== true) {
                    $logger->log('Sicherheit: Ungültige ZIP-Datei abgelehnt: ' . $filename, 'warning');
                    continue;
                }
                $zip->close();
                
                // SICHERHEIT: Path-Traversal-Schutz für Upload-Pfad
                $target_path = $upload_dir . $filename;
                $real_upload_dir = realpath($upload_dir);
                
                if ($real_upload_dir === false) {
                    $logger->log('Sicherheit: Ungültiges Upload-Verzeichnis', 'warning');
                    continue;
                }
                
                // Berechne realen Zielpfad
                $real_target = realpath(dirname($target_path));
                if ($real_target === false) {
                    // Verzeichnis existiert noch nicht, erstelle es
                    $real_target = $real_upload_dir;
                }
                $real_target_path = $real_target . '/' . basename($target_path);
                
                // Path-Traversal-Schutz: Zielpfad muss innerhalb von Upload-Verzeichnis liegen
                if (strpos($real_target_path, $real_upload_dir) !== 0) {
                    $logger->log('Sicherheit: Path Traversal erkannt bei: ' . $filename, 'warning');
                    continue;
                }
                
                if (move_uploaded_file($tmp_name, $real_target_path)) {
                    $uploaded_files[] = [
                        'name' => $filename,
                        'path' => $real_target_path,
                        'size' => filesize($real_target_path),
                    ];
                    $logger->log("Datei hochgeladen: $filename (" . round(filesize($real_target_path) / 1024 / 1024, 2) . " MB)");
                }
            }
            
            if (empty($uploaded_files)) {
                throw new \Exception('Keine Dateien erfolgreich hochgeladen');
            }
            
            $logger->log("Erfolgreich hochgeladen: " . count($uploaded_files) . " von $file_count Datei(en)");
            
            // Prüfen, ob es mehrere zusammengehörige Dateien sind (z.B. UpdraftPlus)
            if (count($uploaded_files) > 1) {
                $logger->log('Mehrere Dateien erkannt, versuche zusammenzuführen...');
                $merge_result = $this->merge_backup_files($uploaded_files);
                
                if ($merge_result['success']) {
                    // Speichere zusammengeführtes Backup
                    global $wpdb;
                    $table_name = $wpdb->prefix . 'jbm_backups';
                    
                    $wpdb->insert($table_name, [
                        'backup_name' => basename($merge_result['merged_path']),
                        'backup_type' => 'merged_upload',
                        'backup_size' => filesize($merge_result['merged_path']),
                        'backup_path' => $merge_result['merged_path'],
                        'backup_status' => 'completed',
                        'backup_files' => json_encode([
                            'source_files' => count($uploaded_files),
                            'merged' => true
                        ]),
                    ]);
                    
                    wp_send_json_success([
                        'message' => sprintf(__('%d Dateien erfolgreich hochgeladen und zusammengeführt', 'jenva-backup-migration'), count($uploaded_files)),
                        'merged' => true,
                        'file_count' => count($uploaded_files),
                    ]);
                } else {
                    // Zusammenführen fehlgeschlagen, speichere einzeln
                    $this->save_individual_backups($uploaded_files);
                    
                    wp_send_json_success([
                        'message' => sprintf(__('%d Dateien hochgeladen (einzeln gespeichert)', 'jenva-backup-migration'), count($uploaded_files)),
                        'merged' => false,
                        'file_count' => count($uploaded_files),
                    ]);
                }
            } else {
                // Einzelne Datei - normale Verarbeitung
                $file = $uploaded_files[0];
                
                // Format erkennen und ggf. konvertieren
                $converter = new \JBM\BackupConverter();
                $detected_type = $converter->detect_backup_type($file['path']);
                
                $backup_type = 'uploaded';
                $final_path = $file['path'];
                $conversion_message = '';
                
                if ($detected_type && $detected_type !== 'unknown') {
                    $conversion_message = " (erkannt als: $detected_type)";
                    
                    if ($detected_type !== 'bmp') {
                        $convert_result = $converter->convert($file['path'], $detected_type);
                        
                        if ($convert_result['success']) {
                            @unlink($file['path']);
                            $final_path = $convert_result['converted_path'];
                            $backup_type = 'converted_' . $detected_type;
                            $conversion_message = " → Automatisch von $detected_type konvertiert";
                        }
                    }
                }
                
                global $wpdb;
                $table_name = $wpdb->prefix . 'jbm_backups';
                
                $wpdb->insert($table_name, [
                    'backup_name' => basename($final_path),
                    'backup_type' => $backup_type,
                    'backup_size' => filesize($final_path),
                    'backup_path' => $final_path,
                    'backup_status' => 'completed',
                    'backup_files' => json_encode(['source' => $detected_type ?: 'unknown']),
                ]);
                
                wp_send_json_success([
                    'message' => __('Backup hochgeladen', 'jenva-backup-migration') . $conversion_message,
                    'detected_type' => $detected_type,
                    'converted' => $detected_type && $detected_type !== 'unknown',
                    'merged' => false,
                ]);
            }
            
        } catch (\Exception $e) {
            $logger->log('Multi-Upload Fehler: ' . $e->getMessage(), 'error');
            wp_send_json_error(['message' => $e->getMessage()]);
        }
    }
    
    /**
     * Führt mehrere Backup-Dateien zusammen (z.B. UpdraftPlus)
     */
    private function merge_backup_files($files) {
        $logger = new \JBM\Logger();
        
        try {
            $merge_dir = JBM_TEMP_DIR . uniqid('merge_') . '/';
            wp_mkdir_p($merge_dir);
            
            $logger->log('Erstelle Zusammenführungs-Verzeichnis: ' . $merge_dir);
            
            // Alle Dateien extrahieren
            foreach ($files as $file) {
                $logger->log('Extrahiere: ' . $file['name']);
                
                $zip = new \ZipArchive();
                if ($zip->open($file['path']) === true) {
                    $zip->extractTo($merge_dir);
                    $zip->close();
                } else {
                    $logger->log('Warnung: Konnte ' . $file['name'] . ' nicht extrahieren', 'warning');
                }
            }
            
            // Erstelle BMP-Manifest
            $manifest = [
                'version' => JBM_VERSION,
                'type' => 'merged',
                'source_files' => count($files),
                'site_url' => get_site_url(),
                'created_at' => current_time('mysql'),
            ];
            file_put_contents($merge_dir . 'manifest.json', json_encode($manifest, JSON_PRETTY_PRINT));
            
            // Erstelle zusammengeführtes ZIP
            $merged_name = 'merged_backup_' . time() . '.zip';
            $merged_path = JBM_BACKUP_DIR . $merged_name;
            
            $zip = new \ZipArchive();
            $zip->open($merged_path, \ZipArchive::CREATE | \ZipArchive::OVERWRITE);
            
            $files_iter = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator($merge_dir, \RecursiveDirectoryIterator::SKIP_DOTS),
                \RecursiveIteratorIterator::LEAVES_ONLY
            );
            
            foreach ($files_iter as $file) {
                if (!$file->isDir()) {
                    $file_path = $file->getRealPath();
                    $relative_path = substr($file_path, strlen($merge_dir));
                    $zip->addFile($file_path, $relative_path);
                }
            }
            
            $zip->close();
            
            // Cleanup
            $this->cleanup_directory($merge_dir);
            
            // Original-Dateien löschen
            foreach ($files as $file) {
                @unlink($file['path']);
            }
            
            $logger->log('Dateien erfolgreich zusammengeführt: ' . $merged_name);
            
            return [
                'success' => true,
                'merged_path' => $merged_path,
            ];
            
        } catch (\Exception $e) {
            $logger->log('Fehler beim Zusammenführen: ' . $e->getMessage(), 'error');
            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }
    
    /**
     * Speichert Dateien einzeln (Fallback)
     */
    private function save_individual_backups($files) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'jbm_backups';
        
        foreach ($files as $file) {
            $wpdb->insert($table_name, [
                'backup_name' => $file['name'],
                'backup_type' => 'uploaded',
                'backup_size' => $file['size'],
                'backup_path' => $file['path'],
                'backup_status' => 'completed',
            ]);
        }
    }
    
    /**
     * Räumt Verzeichnis auf
     */
    private function cleanup_directory($dir) {
        if (!is_dir($dir)) {
            return;
        }
        
        $files = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($dir, \RecursiveDirectoryIterator::SKIP_DOTS),
            \RecursiveIteratorIterator::CHILD_FIRST
        );
        
        foreach ($files as $file) {
            if ($file->isDir()) {
                @rmdir($file->getRealPath());
            } else {
                @unlink($file->getRealPath());
            }
        }
        
        @rmdir($dir);
    }
    
    /**
     * AJAX Handler für Batch-Upload (einzelner Batch)
     */
    public function ajax_upload_batch() {
        check_ajax_referer('jbm_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Keine Berechtigung', 'jenva-backup-migration')]);
        }
        
        $logger = new \JBM\Logger();
        
        // Detaillierte Debug-Informationen
        // Debug-Logging entfernt aus Sicherheitsgründen (keine sensiblen Daten in Logs)
        
        if (empty($_FILES['backup_files'])) {
            $error_msg = 'Keine Dateien in diesem Batch empfangen. $_FILES ist leer oder falsch benannt.';
            $logger->log($error_msg, 'error');
            wp_send_json_error([
                'message' => $error_msg,
                'debug' => [
                    'files_empty' => empty($_FILES),
                    'post_keys' => array_keys($_POST),
                    'files_keys' => array_keys($_FILES),
                ]
            ]);
            return;
        }
        
        $files = $_FILES['backup_files'];
        $file_count = count($files['name']);
        $batch_num = isset($_POST['batch_num']) ? intval($_POST['batch_num']) : 1;
        $total_batches = isset($_POST['total_batches']) ? intval($_POST['total_batches']) : 1;
        
        // Sicherheit: Nur .zip und .jbm Dateien erlauben (Dateiendung + MIME-Type + ZIP-Integrität)
        $allowed_extensions = ['zip', 'jbm'];
        $allowed_mimes = ['application/zip', 'application/x-zip-compressed', 'application/x-zip', 'application/octet-stream'];
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        
        for ($i = 0; $i < $file_count; $i++) {
            // Dateiendung prüfen
            $ext = strtolower(pathinfo($files['name'][$i], PATHINFO_EXTENSION));
            if (!in_array($ext, $allowed_extensions)) {
                finfo_close($finfo);
                wp_send_json_error(['message' => __('Nur ZIP- und JBM-Dateien sind erlaubt', 'jenva-backup-migration')]);
            }
            
            // MIME-Type prüfen
            $mime_type = finfo_file($finfo, $files['tmp_name'][$i]);
            if (!in_array($mime_type, $allowed_mimes)) {
                finfo_close($finfo);
                wp_send_json_error(['message' => __('Ungültiger Dateityp', 'jenva-backup-migration')]);
            }
            
            // ZIP-Integrität prüfen
            $zip = new \ZipArchive();
            if ($zip->open($files['tmp_name'][$i]) !== true) {
                finfo_close($finfo);
                wp_send_json_error(['message' => __('Ungültige oder beschädigte ZIP-Datei', 'jenva-backup-migration')]);
            }
            $zip->close();
        }
        finfo_close($finfo);
        
        $logger->log("Batch-Upload $batch_num/$total_batches gestartet: $file_count Datei(en)");
        
        // Backup-Verzeichnis prüfen
        if (!is_dir(JBM_BACKUP_DIR)) {
            $logger->log('Backup-Verzeichnis existiert nicht, erstelle es...', 'warning');
            wp_mkdir_p(JBM_BACKUP_DIR);
        }
        
        if (!is_writable(JBM_BACKUP_DIR)) {
            $error_msg = 'Backup-Verzeichnis ist nicht beschreibbar: ' . JBM_BACKUP_DIR;
            $logger->log($error_msg, 'error');
            wp_send_json_error([
                'message' => $error_msg . ' | Lösung: chmod 755 oder 775 setzen',
                'backup_dir' => JBM_BACKUP_DIR,
                'writable' => false,
            ]);
            return;
        }
        
        try {
            $upload_dir = JBM_BACKUP_DIR;
            $uploaded_paths = [];
            $errors = [];
            
            // Alle Dateien in diesem Batch hochladen
            for ($i = 0; $i < $file_count; $i++) {
                $filename = $files['name'][$i];
                $file_error = $files['error'][$i];
                $tmp_name = $files['tmp_name'][$i];
                $file_size = $files['size'][$i];
                
                $logger->log("Verarbeite Datei $i: $filename (Größe: " . round($file_size / 1024 / 1024, 2) . " MB, Error-Code: $file_error)");
                
                // Upload-Fehler-Codes interpretieren
                if ($file_error !== UPLOAD_ERR_OK) {
                    $error_messages = [
                        UPLOAD_ERR_INI_SIZE => 'Datei überschreitet upload_max_filesize in php.ini',
                        UPLOAD_ERR_FORM_SIZE => 'Datei überschreitet MAX_FILE_SIZE',
                        UPLOAD_ERR_PARTIAL => 'Datei wurde nur teilweise hochgeladen',
                        UPLOAD_ERR_NO_FILE => 'Keine Datei hochgeladen',
                        UPLOAD_ERR_NO_TMP_DIR => 'Temporäres Verzeichnis fehlt',
                        UPLOAD_ERR_CANT_WRITE => 'Konnte Datei nicht auf Festplatte schreiben',
                        UPLOAD_ERR_EXTENSION => 'Eine PHP-Erweiterung hat den Upload gestoppt',
                    ];
                    
                    $error_detail = isset($error_messages[$file_error]) ? $error_messages[$file_error] : 'Unbekannter Fehler';
                    $logger->log("Upload-Fehler bei $filename: $error_detail (Code: $file_error)", 'error');
                    $errors[] = "$filename: $error_detail";
                    continue;
                }
                
                // Dateinamen bereinigen
                $clean_filename = sanitize_file_name($filename);
                $target_path = $upload_dir . $clean_filename;
                
                // Temporäre Datei prüfen
                if (!file_exists($tmp_name)) {
                    $logger->log("Temporäre Datei nicht gefunden: $tmp_name", 'error');
                    $errors[] = "$filename: Temporäre Datei nicht gefunden";
                    continue;
                }
                
                // Datei verschieben
                if (move_uploaded_file($tmp_name, $target_path)) {
                    $uploaded_paths[] = $target_path;
                    $size_mb = round(filesize($target_path) / 1024 / 1024, 2);
                    $logger->log("✅ Batch $batch_num: $clean_filename hochgeladen ($size_mb MB)");
                } else {
                    $logger->log("❌ Batch $batch_num: Konnte $filename nicht verschieben", 'error');
                    $logger->log("Von: $tmp_name -> Zu: $target_path", 'error');
                    $logger->log("Ziel existiert: " . (file_exists($target_path) ? 'JA' : 'NEIN'), 'error');
                    $logger->log("Ziel beschreibbar: " . (is_writable(dirname($target_path)) ? 'JA' : 'NEIN'), 'error');
                    $errors[] = "$filename: Konnte nicht verschoben werden (Berechtigungsproblem?)";
                }
            }
            
            if (empty($uploaded_paths) && !empty($errors)) {
                $error_summary = "Batch $batch_num fehlgeschlagen. Details:\n" . implode("\n", $errors);
                $logger->log($error_summary, 'error');
                throw new \Exception($error_summary);
            }
            
            if (empty($uploaded_paths)) {
                throw new \Exception("Keine Dateien in Batch $batch_num erfolgreich hochgeladen (unbekannter Grund)");
            }
            
            $success_count = count($uploaded_paths);
            $logger->log("✅ Batch $batch_num/$total_batches abgeschlossen: $success_count von $file_count Datei(en) hochgeladen");
            
            if (!empty($errors)) {
                $logger->log("⚠️ Batch $batch_num hatte " . count($errors) . " Fehler, aber $success_count erfolgreich", 'warning');
            }
            
            wp_send_json_success([
                'message' => sprintf(__('Batch %d/%d: %d von %d Datei(en) hochgeladen', 'jenva-backup-migration'), $batch_num, $total_batches, $success_count, $file_count),
                'uploaded_paths' => $uploaded_paths,
                'batch_num' => $batch_num,
                'total_batches' => $total_batches,
                'errors' => $errors,
                'success_count' => $success_count,
                'total_count' => $file_count,
            ]);
            
        } catch (\Exception $e) {
            $logger->log("Batch-Upload Kritischer Fehler: " . $e->getMessage(), 'error');
            wp_send_json_error([
                'message' => $e->getMessage(),
                'batch_num' => $batch_num,
                'total_batches' => $total_batches,
            ]);
        }
    }
    
    /**
     * AJAX Handler zum Zusammenführen hochgeladener Dateien
     */
    public function ajax_merge_uploaded_files() {
        check_ajax_referer('jbm_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Keine Berechtigung', 'jenva-backup-migration')]);
        }
        
        if (empty($_POST['file_paths'])) {
            wp_send_json_error(['message' => __('Keine Dateien zum Zusammenführen', 'jenva-backup-migration')]);
        }
        
        $file_paths_raw = $_POST['file_paths'];
        if (!is_array($file_paths_raw)) {
            wp_send_json_error(['message' => __('Ungültige Dateipfade', 'jenva-backup-migration')]);
        }
        
        // SICHERHEIT: Validiere alle Dateipfade gegen Path Traversal
        $file_paths = [];
        
        // Stelle sicher, dass Backup-Verzeichnis existiert
        if (!file_exists(JBM_BACKUP_DIR)) {
            wp_mkdir_p(JBM_BACKUP_DIR);
        }
        
        $backup_dir_real = realpath(JBM_BACKUP_DIR);
        if (!$backup_dir_real) {
            // Fallback: Verwende JBM_BACKUP_DIR direkt (normalisiert)
            $backup_dir_real = rtrim(JBM_BACKUP_DIR, '/\\');
            // Normalisiere Pfad-Trenner
            $backup_dir_real = str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $backup_dir_real);
        }
        
        foreach ($file_paths_raw as $file_path_raw) {
            // Sanitize Input
            $file_path_raw = sanitize_text_field($file_path_raw);
            
            // Prüfe auf leere Pfade
            if (empty($file_path_raw)) {
                continue;
            }
            
            // Normalisiere den Pfad (entfernt .. und normalisiert)
            $normalized_path = realpath($file_path_raw);
            
            // Prüfe ob Datei existiert
            if ($normalized_path === false || !file_exists($normalized_path)) {
                $logger = new \JBM\Logger();
                $logger->log('Warnung: Datei nicht gefunden oder ungültiger Pfad: ' . esc_html($file_path_raw), 'warning');
                continue;
            }
            
            // KRITISCH: Prüfe ob Pfad innerhalb des Backup-Verzeichnisses liegt (Path Traversal Prevention)
            // Verwendung von strpos mit strikter Längenprüfung
            $backup_dir_len = strlen($backup_dir_real);
            if (strlen($normalized_path) < $backup_dir_len || 
                substr($normalized_path, 0, $backup_dir_len) !== $backup_dir_real ||
                (strlen($normalized_path) > $backup_dir_len && $normalized_path[$backup_dir_len] !== DIRECTORY_SEPARATOR)) {
                // Pfad liegt außerhalb des Backup-Verzeichnisses - SICHERHEITSLÜCKE verhindert!
                $logger = new \JBM\Logger();
                $logger->log('SICHERHEIT: Versuch auf Datei außerhalb des Backup-Verzeichnisses blockiert: ' . esc_html($file_path_raw), 'error');
                continue;
            }
            
            // Pfad ist sicher - hinzufügen
            $file_paths[] = $normalized_path;
        }
        
        if (empty($file_paths)) {
            wp_send_json_error(['message' => __('Keine gültigen Dateien zum Zusammenführen gefunden', 'jenva-backup-migration')]);
        }
        
        $file_count = count($file_paths);
        
        $logger = new \JBM\Logger();
        $logger->log("Starte Zusammenführung von $file_count Dateien...");
        
        try {
            @set_time_limit(0);
            @ini_set('memory_limit', '1024M');
            
            $merge_dir = JBM_TEMP_DIR . uniqid('merge_') . '/';
            wp_mkdir_p($merge_dir);
            
            $logger->log('Erstelle Zusammenführungs-Verzeichnis: ' . $merge_dir);
            
            // Extract all files / Alle Dateien extrahieren
            $extracted_count = 0;
            foreach ($file_paths as $file_path) {
                // Additional security check (should already be validated, but Defense in Depth) / Zusätzliche Sicherheitsprüfung (sollte bereits validiert sein, aber Defense in Depth)
                if (!file_exists($file_path)) {
                    $logger->log('Warnung: Datei nicht gefunden: ' . basename($file_path), 'warning');
                    continue;
                }
                
                // Re-validate path (Defense in Depth) / Nochmalige Pfad-Validierung (Defense in Depth)
                $file_path_real = realpath($file_path);
                if ($file_path_real === false || 
                    substr($file_path_real, 0, strlen($backup_dir_real)) !== $backup_dir_real) {
                    $logger->log('SICHERHEIT: Pfad-Validierung fehlgeschlagen: ' . basename($file_path), 'error');
                    continue;
                }
                
                $logger->log('Extrahiere: ' . basename($file_path));
                
                $zip = new \ZipArchive();
                if ($zip->open($file_path) === true) {
                    $zip->extractTo($merge_dir);
                    $zip->close();
                    $extracted_count++;
                    $logger->log('Erfolgreich extrahiert: ' . basename($file_path));
                } else {
                    $logger->log('Warnung: Konnte nicht extrahieren: ' . basename($file_path), 'warning');
                }
            }
            
            if ($extracted_count == 0) {
                throw new \Exception('Keine Dateien konnten extrahiert werden');
            }
            
            $logger->log("$extracted_count von $file_count Dateien extrahiert");
            
            // Erstelle BMP-Manifest
            $manifest = [
                'version' => JBM_VERSION,
                'type' => 'merged',
                'source_files' => $file_count,
                'site_url' => get_site_url(),
                'home_url' => get_home_url(),
                'created_at' => current_time('mysql'),
                'merged_at' => current_time('mysql'),
            ];
            file_put_contents($merge_dir . 'manifest.json', json_encode($manifest, JSON_PRETTY_PRINT));
            
            // Erstelle zusammengeführtes ZIP
            $logger->log('Erstelle zusammengeführtes ZIP-Archiv...');
            $merged_name = 'merged_updraft_' . time() . '.zip';
            $merged_path = JBM_BACKUP_DIR . $merged_name;
            
            $zip = new \ZipArchive();
            if ($zip->open($merged_path, \ZipArchive::CREATE | \ZipArchive::OVERWRITE) !== true) {
                throw new \Exception('Konnte zusammengeführtes ZIP nicht erstellen');
            }
            
            // Kompression auf Level 9 setzen
            $files_iter = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator($merge_dir, \RecursiveDirectoryIterator::SKIP_DOTS),
                \RecursiveIteratorIterator::LEAVES_ONLY
            );
            
            $added_files = 0;
            foreach ($files_iter as $file) {
                if (!$file->isDir()) {
                    $file_path = $file->getRealPath();
                    $relative_path = substr($file_path, strlen($merge_dir));
                    $zip->addFile($file_path, $relative_path);
                    
                    // Kompression setzen
                    if (preg_match('/\.(sql|txt|log|json|xml|html|css|js)$/i', $file_path)) {
                        $zip->setCompressionName($relative_path, \ZipArchive::CM_DEFLATE, 9);
                    }
                    
                    $added_files++;
                }
            }
            
            $zip->close();
            
            $final_size = filesize($merged_path);
            $logger->log("Zusammengeführtes ZIP erstellt: $added_files Dateien, " . round($final_size / 1024 / 1024, 2) . " MB");
            
            // Cleanup
            $logger->log('Räume temporäre Dateien auf...');
            $this->cleanup_directory($merge_dir);
            
            // Original-Dateien löschen
            foreach ($file_paths as $file_path) {
                if (file_exists($file_path)) {
                    @unlink($file_path);
                }
            }
            
            // In Datenbank speichern
            global $wpdb;
            $table_name = $wpdb->prefix . 'jbm_backups';
            
            $wpdb->insert($table_name, [
                'backup_name' => $merged_name,
                'backup_type' => 'merged_updraft',
                'backup_size' => $final_size,
                'backup_path' => $merged_path,
                'backup_status' => 'completed',
                'backup_files' => json_encode([
                    'source_files' => $file_count,
                    'merged' => true,
                    'merged_files' => $added_files,
                ]),
            ]);
            
            $logger->log("Zusammenführung abgeschlossen! Backup-ID: " . $wpdb->insert_id);
            
            wp_send_json_success([
                'message' => sprintf(__('%d Dateien erfolgreich zusammengeführt', 'jenva-backup-migration'), $file_count),
                'merged_path' => $merged_path,
                'merged_size' => $final_size,
                'file_count' => $file_count,
            ]);
            
        } catch (\Exception $e) {
            $logger->log('Merge-Fehler: ' . $e->getMessage(), 'error');
            wp_send_json_error(['message' => 'Fehler beim Zusammenführen: ' . $e->getMessage()]);
        }
    }
    
    /**
     * AJAX: Holt die letzten N Log-Einträge für Live-Anzeige
     */
    public function ajax_get_live_logs() {
        check_ajax_referer('jbm_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Keine Berechtigung', 'jenva-backup-migration')]);
        }
        
        $count = isset($_POST['count']) ? intval($_POST['count']) : 6;
        $log_file = JBM_LOG_DIR . 'backup-migration.log';
        
        if (!file_exists($log_file)) {
            wp_send_json_success(['logs' => []]);
            return;
        }
        
        try {
            // Read last N lines from log / Lese die letzten N Zeilen aus dem Log
            $logs = [];
            $file = new \SplFileObject($log_file, 'r');
            $file->seek(PHP_INT_MAX);
            $total_lines = $file->key() + 1;
            
            $start_line = max(0, $total_lines - $count);
            $file->seek($start_line);
            
            while (!$file->eof()) {
                $line = trim($file->fgets());
                if (!empty($line)) {
                    $logs[] = $line;
                }
            }
            
            wp_send_json_success(['logs' => array_slice($logs, -$count)]);
            
        } catch (\Exception $e) {
            wp_send_json_error(['message' => $e->getMessage()]);
        }
    }
    
    public function ajax_migrate_site() {
        check_ajax_referer('jbm_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Keine Berechtigung', 'jenva-backup-migration')]);
        }
        
        // Sicherheit: URLs sanitieren und validieren
        $old_url = isset($_POST['old_url']) ? esc_url_raw($_POST['old_url']) : '';
        $new_url = isset($_POST['new_url']) ? esc_url_raw($_POST['new_url']) : '';
        
        // Validierung: Prüfe ob URLs leer sind
        if (empty($old_url)) {
            wp_send_json_error(['message' => __('Die alte URL darf nicht leer sein', 'jenva-backup-migration')]);
            return;
        }
        if (empty($new_url)) {
            wp_send_json_error(['message' => __('Die neue URL darf nicht leer sein', 'jenva-backup-migration')]);
            return;
        }
        
        // Sicherheit: URLs mit filter_var validieren
        if (!filter_var($old_url, FILTER_VALIDATE_URL)) {
            wp_send_json_error(['message' => __('Die alte URL ist ungültig', 'jenva-backup-migration')]);
            return;
        }
        if (!filter_var($new_url, FILTER_VALIDATE_URL)) {
            wp_send_json_error(['message' => __('Die neue URL ist ungültig', 'jenva-backup-migration')]);
            return;
        }
        
        // Sicherheit: Maximale URL-Länge prüfen (verhindert DoS durch überlange Strings)
        if (strlen($old_url) > 2000 || strlen($new_url) > 2000) {
            wp_send_json_error(['message' => __('URL zu lang (max. 2000 Zeichen)', 'jenva-backup-migration')]);
            return;
        }
        
        // Sicherheit: Nur HTTP/HTTPS erlauben
        $old_scheme = parse_url($old_url, PHP_URL_SCHEME);
        $new_scheme = parse_url($new_url, PHP_URL_SCHEME);
        if (!in_array($old_scheme, ['http', 'https'], true) || !in_array($new_scheme, ['http', 'https'], true)) {
            wp_send_json_error(['message' => __('Nur HTTP und HTTPS URLs sind erlaubt', 'jenva-backup-migration')]);
            return;
        }
        
        $migration = new \JBM\Migration();
        $result = $migration->migrate($old_url, $new_url);
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
    
    public function ajax_save_schedule() {
        check_ajax_referer('jbm_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Keine Berechtigung', 'jenva-backup-migration')]);
        }
        
        // Welcher Zeitplan? (1 oder 2)
        $schedule_num = isset($_POST['schedule_num']) ? intval($_POST['schedule_num']) : 1;
        
        $schedule_data = [
            'enabled' => isset($_POST['enabled']) ? (bool)$_POST['enabled'] : false,
            'frequency' => isset($_POST['frequency']) ? sanitize_text_field($_POST['frequency']) : 'daily',
            'time' => isset($_POST['time']) ? sanitize_text_field($_POST['time']) : '02:00',
            'weekday' => isset($_POST['weekday']) ? sanitize_text_field($_POST['weekday']) : '1',
            'monthday' => isset($_POST['monthday']) ? sanitize_text_field($_POST['monthday']) : '1',
            'backup_type' => isset($_POST['backup_type']) ? sanitize_text_field($_POST['backup_type']) : 'full',
        ];
        
        // In separater Option speichern
        update_option('jbm_schedule_' . $schedule_num, $schedule_data);
        
        // Letzte Ausführung auf jetzt setzen (verhindert sofortiges Triggern)
        update_option('jbm_schedule_' . $schedule_num . '_last_run', time());
        
        // Logger
        $logger = new \JBM\Logger();
        $logger->log("Zeitplan {$schedule_num} wird gespeichert: " . json_encode($schedule_data));
        
        $schedule = new \JBM\Schedule();
        $result = $schedule->update_schedule($schedule_num, $schedule_data);
        
        if ($result['success']) {
            $logger->log("Zeitplan {$schedule_num} erfolgreich gespeichert - Nächstes Backup: " . ($result['next_run'] ?? 'N/A'));
        } else {
            $logger->log("Fehler beim Speichern von Zeitplan {$schedule_num}: " . ($result['message'] ?? 'Unbekannt'), 'error');
        }
        
        if ($result['success']) {
            wp_send_json_success([
                'message' => sprintf(__('Zeitplan %d gespeichert', 'jenva-backup-migration'), $schedule_num) . 
                    (isset($result['next_run']) ? ' - Nächstes Backup: ' . $result['next_run'] : '')
            ]);
        } else {
            wp_send_json_error(['message' => $result['message'] ?? 'Unbekannter Fehler']);
        }
    }
    
    public function ajax_get_backups() {
        check_ajax_referer('jbm_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Keine Berechtigung', 'jenva-backup-migration')]);
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'jbm_backups';
        $backups = $wpdb->get_results("SELECT * FROM $table_name ORDER BY backup_date DESC");
        
        wp_send_json_success(['backups' => $backups]);
    }
    
    // Admin-Seiten rendern
    public function render_main_page() {
        $this->ensure_database_exists();
        include JBM_PLUGIN_DIR . 'templates/admin-main.php';
    }
    
    public function render_schedule_page() {
        $this->ensure_database_exists();
        include JBM_PLUGIN_DIR . 'templates/admin-schedule.php';
    }
    
    public function render_settings_page() {
        $this->ensure_database_exists();
        include JBM_PLUGIN_DIR . 'templates/admin-settings.php';
    }
    
    /**
     * Stellt sicher, dass die Datenbanktabelle existiert
     */
    private function ensure_database_exists() {
        // Nur einmal pro Request prüfen
        if ($this->db_checked) {
            return;
        }
        
        $this->db_checked = true;
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'jbm_backups';
        
        // Prüfen, ob Tabelle existiert
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '$table_name'") === $table_name;
        
        if (!$table_exists) {
            // Tabelle erstellen
            $success = $this->create_database_table();
            $this->create_directories();
            
            if ($success) {
                // Erfolgsmeldung anzeigen
                add_action('admin_notices', function() {
                    ?>
                    <div class="notice notice-success is-dismissible">
                        <p><strong>Jenva Jenva Backup & Migration:</strong> Datenbanktabelle wurde automatisch erstellt! Sie können jetzt Backups erstellen. 🎉</p>
                    </div>
                    <?php
                });
            } else {
                // Fehlermeldung anzeigen
                add_action('admin_notices', function() {
                    ?>
                    <div class="notice notice-error">
                        <p><strong>Jenva Jenva Backup & Migration:</strong> Datenbanktabelle konnte nicht erstellt werden. Bitte prüfen Sie die Datenbankrechte.</p>
                    </div>
                    <?php
                });
            }
        } else {
            // Tabelle existiert - prüfe ob 'protected' Spalte existiert
            $columns = $wpdb->get_results("SHOW COLUMNS FROM $table_name");
            $has_protected = false;
            
            foreach ($columns as $column) {
                if ($column->Field === 'protected') {
                    $has_protected = true;
                    break;
                }
            }
            
            // Falls 'protected' Spalte fehlt, hinzufügen
            if (!$has_protected) {
                $wpdb->query("ALTER TABLE $table_name ADD protected tinyint(1) DEFAULT 0 AFTER backup_files");
                $wpdb->query("ALTER TABLE $table_name ADD INDEX protected (protected)");
            }
        }
    }
    
    /**
     * AJAX: Backup-Schutz umschalten
     */
    public function ajax_toggle_protection() {
        check_ajax_referer('jbm_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Keine Berechtigung', 'jenva-backup-migration')]);
        }
        
        $backup_id = isset($_POST['backup_id']) ? intval($_POST['backup_id']) : 0;
        
        if (!$backup_id) {
            wp_send_json_error(['message' => __('Ungültige Backup-ID', 'jenva-backup-migration')]);
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'jbm_backups';
        
        // Aktuellen Schutzstatus holen
        $current_status = $wpdb->get_var($wpdb->prepare(
            "SELECT protected FROM $table_name WHERE id = %d",
            $backup_id
        ));
        
        // Umschalten
        $new_status = $current_status ? 0 : 1;
        
        $result = $wpdb->update(
            $table_name,
            ['protected' => $new_status],
            ['id' => $backup_id],
            ['%d'],
            ['%d']
        );
        
        if ($result !== false) {
            $message = $new_status ? 
                __('Backup ist jetzt geschützt und wird nicht automatisch gelöscht', 'jenva-backup-migration') :
                __('Backup-Schutz aufgehoben', 'jenva-backup-migration');
                
            wp_send_json_success([
                'message' => $message,
                'protected' => $new_status
            ]);
        } else {
            wp_send_json_error(['message' => __('Fehler beim Aktualisieren', 'jenva-backup-migration')]);
        }
    }
    
    /**
     * AJAX: Alte Backups basierend auf Limits löschen
     */
    public function ajax_cleanup_old_backups() {
        check_ajax_referer('jbm_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Keine Berechtigung', 'jenva-backup-migration')]);
        }
        
        $result = $this->cleanup_old_backups();
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
    
    /**
     * AJAX: Testet die Zeitplan-Ausführung manuell
     */
    public function ajax_test_schedule() {
        check_ajax_referer('jbm_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Keine Berechtigung', 'jenva-backup-migration')]);
        }
        
        $schedule_num = isset($_POST['schedule_num']) ? intval($_POST['schedule_num']) : 1;
        
        $logger = new \JBM\Logger();
        $logger->log("=====================================");
        $logger->log("🧪 MANUELLER TEST GESTARTET: Zeitplan {$schedule_num}");
        $logger->log("=====================================");
        
        // Lade Zeitplan-Einstellungen
        $schedule_data = get_option('jbm_schedule_' . $schedule_num);
        $logger->log("Zeitplan-Daten geladen: " . json_encode($schedule_data));
        
        if (!$schedule_data || empty($schedule_data['enabled'])) {
            $logger->log("FEHLER: Zeitplan {$schedule_num} ist deaktiviert oder nicht konfiguriert", 'error');
            wp_send_json_error(['message' => "Zeitplan {$schedule_num} ist nicht aktiviert oder nicht konfiguriert"]);
            return;
        }
        
        $logger->log("Zeitplan ist aktiv, starte Backup-Erstellung...");
        $logger->log("Backup-Typ: " . $schedule_data['backup_type']);
        
        try {
            // Direkt Backup erstellen (ohne Schedule-Klasse dazwischen)
            $backup = new \JBM\Backup();
            $logger->log("Backup-Klasse erstellt, rufe create() auf...");
            
            $result = $backup->create($schedule_data['backup_type'], "scheduled-{$schedule_num}");
            
            $logger->log("Backup::create() returned: " . json_encode($result));
            
            if ($result && isset($result['success']) && $result['success']) {
                $logger->log("✅ TEST ERFOLGREICH: " . ($result['backup_name'] ?? 'N/A'));
                
                // Prüfe ob Datei wirklich existiert
                $backup_path = isset($result['backup_path']) ? $result['backup_path'] : (JBM_BACKUP_DIR . $result['backup_name']);
                $file_exists = file_exists($backup_path);
                $file_size = $file_exists ? filesize($backup_path) : 0;
                
                $logger->log("Datei existiert: " . ($file_exists ? 'JA' : 'NEIN'));
                $logger->log("Dateipfad: " . $backup_path);
                $logger->log("Dateigröße: " . size_format($file_size));
                
                // Prüfe DB-Eintrag
                global $wpdb;
                $table_name = $wpdb->prefix . 'jbm_backups';
                $db_entry = $wpdb->get_row($wpdb->prepare(
                    "SELECT * FROM $table_name WHERE backup_name = %s ORDER BY id DESC LIMIT 1",
                    $result['backup_name']
                ));
                
                $logger->log("DB-Eintrag vorhanden: " . ($db_entry ? 'JA (ID: ' . $db_entry->id . ')' : 'NEIN'));
                
                wp_send_json_success([
                    'message' => "Zeitplan {$schedule_num} erfolgreich ausgeführt!\n\nBackup: " . ($result['backup_name'] ?? 'N/A') . "\nGröße: " . size_format($file_size) . "\nDatei existiert: " . ($file_exists ? 'Ja' : 'NEIN!') . "\nDB-Eintrag: " . ($db_entry ? 'Ja (ID: ' . $db_entry->id . ')' : 'NEIN!'),
                    'debug' => [
                        'file_exists' => $file_exists,
                        'file_path' => $backup_path,
                        'db_entry' => $db_entry ? true : false,
                        'result' => $result
                    ]
                ]);
            } else {
                $logger->log("❌ TEST FEHLGESCHLAGEN: " . ($result['message'] ?? 'Unbekannter Fehler'), 'error');
                wp_send_json_error([
                    'message' => "Zeitplan {$schedule_num} fehlgeschlagen!\n\nFehler: " . ($result['message'] ?? 'Unbekannter Fehler')
                ]);
            }
            
        } catch (\Exception $e) {
            $logger->log("❌ KRITISCHER FEHLER: " . $e->getMessage(), 'error');
            $logger->log("Stack Trace: " . $e->getTraceAsString(), 'error');
            wp_send_json_error(['message' => 'Kritischer Fehler: ' . $e->getMessage() . "\n\nDatei: " . basename($e->getFile()) . ':' . $e->getLine()]);
        }
    }
    
    /**
     * AJAX: Lädt einen einzelnen Chunk einer Datei hoch
     * Umgeht 413 Request Entity Too Large durch Chunking
     * Sicherheit: Umfassende Validierung von Dateinamen und Inhalten
     */
    public function ajax_upload_chunk() {
        check_ajax_referer('jbm_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Keine Berechtigung', 'jenva-backup-migration')]);
        }
        
        if (!isset($_FILES['file_chunk'])) {
            wp_send_json_error(['message' => __('Kein Chunk empfangen', 'jenva-backup-migration')]);
        }
        
        try {
            $chunk = $_FILES['file_chunk'];
            
            // Sicherheit: Dateinamen validieren und bereinigen
            $raw_file_name = isset($_POST['file_name']) ? $_POST['file_name'] : '';
            $file_name = $this->validate_upload_filename($raw_file_name);
            
            if ($file_name === false) {
                wp_send_json_error(['message' => __('Ungültiger Dateiname', 'jenva-backup-migration')]);
                return;
            }
            
            $chunk_index = intval($_POST['chunk_index']);
            $total_chunks = intval($_POST['total_chunks']);
            $file_size = intval($_POST['file_size']);
            
            // Rate Limiting: Max 100 Chunk-Uploads pro Minute (schützt vor DoS bei vielen parallelen Uploads)
            // Nur beim ersten Chunk einer Datei prüfen
            if ($chunk_index === 0) {
                $rate_check = $this->check_rate_limit('upload_start', 5, 60); // Max 5 neue Uploads pro Minute
                if (is_array($rate_check) && isset($rate_check['limited'])) {
                    wp_send_json_error([
                        'message' => $rate_check['message'],
                        'rate_limited' => true,
                        'wait_time' => $rate_check['wait'],
                    ]);
                    return;
                }
            }
            
            // Sicherheit: Validiere numerische Parameter
            if ($chunk_index < 0 || $total_chunks < 1 || $chunk_index >= $total_chunks) {
                wp_send_json_error(['message' => __('Ungültige Chunk-Parameter', 'jenva-backup-migration')]);
                return;
            }
            
            // Sicherheit: Maximale Dateigröße prüfen (10 GB)
            $max_file_size = 10 * 1024 * 1024 * 1024; // 10 GB
            if ($file_size > $max_file_size) {
                wp_send_json_error(['message' => __('Datei zu groß (max. 10 GB)', 'jenva-backup-migration')]);
                return;
            }
            
            // Sicherheit: Maximale Chunk-Anzahl prüfen (verhindert DoS)
            $max_chunks = 20000; // 20.000 Chunks à 1MB = 20 GB max
            if ($total_chunks > $max_chunks) {
                wp_send_json_error(['message' => __('Zu viele Chunks', 'jenva-backup-migration')]);
                return;
            }
            
            // Sicherheit: Erlaubte Dateiendungen (Whitelist)
            $allowed_extensions = ['zip', 'jbm', 'wpress', 'gz'];
            $ext = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
            if (!in_array($ext, $allowed_extensions, true)) {
                wp_send_json_error(['message' => __('Nur ZIP-, JBM-, WPRESS- und GZ-Dateien sind erlaubt', 'jenva-backup-migration')]);
                return;
            }
            
            // Sicherheit: MIME-Type / Magic Bytes beim ERSTEN Chunk validieren
            if ($chunk_index === 0) {
                $magic_valid = $this->validate_chunk_magic_bytes($chunk['tmp_name'], $ext);
                if (!$magic_valid) {
                    wp_send_json_error([
                        'message' => __('Ungültiges Dateiformat: Die Datei entspricht nicht dem erwarteten Typ.', 'jenva-backup-migration')
                    ]);
                    return;
                }
            }
            
            // Sicherheit: Upload-Fehler prüfen
            if ($chunk['error'] !== UPLOAD_ERR_OK) {
                $error_messages = [
                    UPLOAD_ERR_INI_SIZE => 'Datei überschreitet upload_max_filesize',
                    UPLOAD_ERR_FORM_SIZE => 'Datei überschreitet MAX_FILE_SIZE',
                    UPLOAD_ERR_PARTIAL => 'Datei nur teilweise hochgeladen',
                    UPLOAD_ERR_NO_FILE => 'Keine Datei hochgeladen',
                    UPLOAD_ERR_NO_TMP_DIR => 'Temporäres Verzeichnis fehlt',
                    UPLOAD_ERR_CANT_WRITE => 'Schreibfehler',
                    UPLOAD_ERR_EXTENSION => 'Durch Erweiterung gestoppt',
                ];
                $error_msg = isset($error_messages[$chunk['error']]) ? $error_messages[$chunk['error']] : 'Unbekannter Upload-Fehler';
                wp_send_json_error(['message' => $error_msg]);
                return;
            }
            
            // Temporäres Verzeichnis für Chunks (mit sicherem Hash)
            $safe_hash = hash('sha256', $file_name . wp_get_session_token());
            $chunks_dir = JBM_TEMP_DIR . 'chunks_' . substr($safe_hash, 0, 32) . '/';
            
            // Sicherheit: Stelle sicher, dass Verzeichnis innerhalb von JBM_TEMP_DIR liegt
            $real_temp_dir = realpath(JBM_TEMP_DIR);
            if (!file_exists($chunks_dir)) {
                wp_mkdir_p($chunks_dir);
            }
            
            $real_chunks_dir = realpath($chunks_dir);
            if ($real_chunks_dir === false || strpos($real_chunks_dir, $real_temp_dir) !== 0) {
                wp_send_json_error(['message' => __('Sicherheitsfehler: Ungültiges Verzeichnis', 'jenva-backup-migration')]);
                return;
            }
            
            // Chunk speichern (mit sicherem Pfad)
            $chunk_path = $real_chunks_dir . '/chunk_' . $chunk_index;
            if (!move_uploaded_file($chunk['tmp_name'], $chunk_path)) {
                throw new \Exception('Konnte Chunk nicht speichern');
            }
            
            // Prüfe ob alle Chunks da sind
            if ($chunk_index + 1 == $total_chunks) {
                // Letzter Chunk - jetzt zusammenfügen!
                $logger = new \JBM\Logger();
                $logger->log("🔗 Füge {$total_chunks} Chunks zusammen: {$file_name}");
                
                // Sicherheit: Finaler Pfad validieren (Path-Traversal-Schutz)
                $real_backup_dir = realpath(JBM_BACKUP_DIR);
                if ($real_backup_dir === false) {
                    throw new \Exception('Sicherheitsfehler: Backup-Verzeichnis ungültig');
                }
                
                // Verwende nur den Dateinamen (verhindert Path Traversal)
                $safe_filename = basename($file_name);
                $final_path = $real_backup_dir . '/' . $safe_filename;
                
                // Berechne realen Pfad und prüfe nochmal
                $real_final_path = realpath(dirname($final_path));
                if ($real_final_path === false) {
                    $real_final_path = $real_backup_dir;
                }
                $real_final_path = $real_final_path . '/' . basename($final_path);
                
                // Path-Traversal-Schutz: Finaler Pfad muss innerhalb von Backup-Verzeichnis liegen
                if (strpos($real_final_path, $real_backup_dir) !== 0) {
                    throw new \Exception('Sicherheitsfehler: Path Traversal erkannt');
                }
                
                // Verwende den validierten Pfad
                $final_path = $real_final_path;
                
                $final_file = fopen($final_path, 'wb');
                
                if (!$final_file) {
                    throw new \Exception('Konnte Zieldatei nicht erstellen');
                }
                
                // Alle Chunks in richtiger Reihenfolge zusammenfügen (speichereffizient)
                for ($i = 0; $i < $total_chunks; $i++) {
                    // SICHERHEIT: Chunk-Pfad validieren
                    $chunk_file = $real_chunks_dir . '/chunk_' . $i;
                    $real_chunk_file = realpath($chunk_file);
                    
                    // Path-Traversal-Schutz: Chunk muss innerhalb von Chunks-Verzeichnis liegen
                    if ($real_chunk_file === false || strpos($real_chunk_file, $real_chunks_dir) !== 0) {
                        fclose($final_file);
                        @unlink($final_path); // Lösche unvollständige Datei
                        throw new \Exception("Sicherheitsfehler: Ungültiger Chunk-Pfad {$i}");
                    }
                    
                    if (!file_exists($real_chunk_file)) {
                        fclose($final_file);
                        @unlink($final_path); // Lösche unvollständige Datei
                        throw new \Exception("Chunk {$i} fehlt!");
                    }
                    
                    // Stream-basiertes Kopieren (speichereffizient für große Dateien)
                    $chunk_handle = fopen($real_chunk_file, 'rb');
                    if ($chunk_handle) {
                        while (!feof($chunk_handle)) {
                            $buffer = fread($chunk_handle, 1024 * 1024); // 1MB Buffer
                            if ($buffer !== false) {
                                fwrite($final_file, $buffer);
                            }
                            unset($buffer); // Speicher sofort freigeben
                        }
                        fclose($chunk_handle);
                    }
                    unlink($real_chunk_file); // Chunk löschen nach Verwendung
                    
                    // Memory freigeben nach jedem Chunk
                    if (function_exists('gc_mem_caches')) {
                        gc_mem_caches();
                    }
                }
                
                fclose($final_file);
                
                // Chunks-Verzeichnis aufräumen
                @rmdir($real_chunks_dir);
                
                // Prüfe finale Datei
                $final_size = filesize($final_path);
                $logger->log("✅ Datei zusammengefügt: {$file_name} (" . size_format($final_size) . ")");
                
                // Sicherheit: Validiere ZIP/JBM-Datei (beide sind ZIP-Archive)
                if (in_array($ext, ['zip', 'jbm']) && class_exists('ZipArchive')) {
                    $zip = new \ZipArchive();
                    $zip_result = $zip->open($final_path, \ZipArchive::CHECKCONS);
                    if ($zip_result !== true) {
                        @unlink($final_path);
                        throw new \Exception('Hochgeladene ' . strtoupper($ext) . '-Datei ist beschädigt oder ungültig');
                    }
                    
                    // Sicherheit: Prüfe auf gefährliche Dateien im ZIP
                    $dangerous_found = false;
                    for ($i = 0; $i < $zip->numFiles; $i++) {
                        $zip_filename = $zip->getNameIndex($i);
                        // Prüfe auf PHP-Dateien oder andere ausführbare Dateien außerhalb des erwarteten Kontexts
                        // Hinweis: Backup-ZIPs dürfen PHP-Dateien enthalten (Plugins/Themes)
                        // Wir prüfen hier nur auf offensichtliche Angriffsmuster
                        if (preg_match('/\.(phar|htaccess|htpasswd)$/i', $zip_filename)) {
                            $dangerous_found = true;
                            $logger->log("⚠️ Warnung: Potentiell gefährliche Datei im Upload: {$zip_filename}", 'warning');
                        }
                    }
                    $zip->close();
                }
                
                // In Datenbank speichern
                global $wpdb;
                $table_name = $wpdb->prefix . 'jbm_backups';
                
                $wpdb->insert($table_name, [
                    'backup_name' => $file_name,
                    'backup_type' => 'uploaded',
                    'backup_size' => $final_size,
                    'backup_path' => $final_path,
                    'backup_status' => 'completed',
                    'backup_files' => json_encode(['uploaded' => true, 'chunked' => true]),
                ]);
                
                $logger->log("💾 In Datenbank gespeichert (ID: " . $wpdb->insert_id . ")");
                
                wp_send_json_success([
                    'message' => 'Datei erfolgreich hochgeladen',
                    'file_path' => $final_path,
                    'complete' => true
                ]);
            } else {
                // Chunk gespeichert, warte auf weitere
                wp_send_json_success([
                    'message' => "Chunk {$chunk_index} gespeichert",
                    'complete' => false
                ]);
            }
            
        } catch (\Exception $e) {
            wp_send_json_error(['message' => $e->getMessage()]);
        }
    }
    
    /**
     * AJAX: Lädt eine einzelne Backup-Datei hoch (sequenziell)
     */
    public function ajax_upload_single_file() {
        check_ajax_referer('jbm_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Keine Berechtigung', 'jenva-backup-migration')]);
        }
        
        if (!isset($_FILES['backup_file'])) {
            wp_send_json_error(['message' => __('Keine Datei hochgeladen', 'jenva-backup-migration')]);
        }
        
        $logger = new \JBM\Logger();
        
        $file = $_FILES['backup_file'];
        $file_num = isset($_POST['file_num']) ? intval($_POST['file_num']) : 1;
        $total_files = isset($_POST['total_files']) ? intval($_POST['total_files']) : 1;
        
        $logger->log("📤 Sequenzieller Upload: Datei {$file_num}/{$total_files}: " . $file['name']);
        
        try {
            // Prüfe Upload-Fehler
            if ($file['error'] !== UPLOAD_ERR_OK) {
                throw new \Exception('Upload-Fehler: ' . $file['error']);
            }
            
            $filename = sanitize_file_name($file['name']);
            
            // Sicherheit: Nur .zip und .jbm Dateien erlauben (Dateiendung)
            $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
            if (!in_array($ext, ['zip', 'jbm'])) {
                throw new \Exception(__('Nur ZIP- und JBM-Dateien sind erlaubt', 'jenva-backup-migration'));
            }
            
            // Sicherheit: MIME-Type Validierung
            $allowed_mimes = ['application/zip', 'application/x-zip-compressed', 'application/x-zip', 'application/octet-stream'];
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            $mime_type = finfo_file($finfo, $file['tmp_name']);
            finfo_close($finfo);
            
            if (!in_array($mime_type, $allowed_mimes)) {
                $logger->log("Sicherheit: Ungültiger MIME-Type: {$mime_type}", 'error');
                throw new \Exception(__('Ungültiger Dateityp', 'jenva-backup-migration'));
            }
            
            // Sicherheit: ZIP-Integrität prüfen
            $zip = new \ZipArchive();
            if ($zip->open($file['tmp_name']) !== true) {
                throw new \Exception(__('Ungültige oder beschädigte ZIP-Datei', 'jenva-backup-migration'));
            }
            $zip->close();
            
            $target_path = JBM_BACKUP_DIR . $filename;
            
            // Datei verschieben
            if (!move_uploaded_file($file['tmp_name'], $target_path)) {
                throw new \Exception('Konnte Datei nicht verschieben');
            }
            
            $file_size = filesize($target_path);
            $logger->log("  ✅ Hochgeladen: {$filename} (" . size_format($file_size) . ")");
            
            // In Datenbank speichern (nur wenn es die letzte Datei ist ODER eine einzelne Datei)
            if ($file_num == $total_files) {
                global $wpdb;
                $table_name = $wpdb->prefix . 'jbm_backups';
                
                $wpdb->insert($table_name, [
                    'backup_name' => $filename,
                    'backup_type' => 'uploaded',
                    'backup_size' => $file_size,
                    'backup_path' => $target_path,
                    'backup_status' => 'completed',
                    'backup_files' => json_encode(['uploaded' => true]),
                ]);
                
                $logger->log("  💾 In Datenbank gespeichert (ID: " . $wpdb->insert_id . ")");
            }
            
            wp_send_json_success([
                'message' => "Datei {$file_num}/{$total_files} hochgeladen",
                'file_path' => $target_path,
                'file_name' => $filename,
                'file_size' => $file_size
            ]);
            
        } catch (\Exception $e) {
            $logger->log("  ❌ Upload-Fehler: " . $e->getMessage(), 'error');
            wp_send_json_error(['message' => $e->getMessage()]);
        }
    }
    
    /**
     * AJAX: Diagnose-Test für Backup-Funktionalität
     */
    public function ajax_diagnose_backup() {
        check_ajax_referer('jbm_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Keine Berechtigung', 'jenva-backup-migration')]);
        }
        
        $logger = new \JBM\Logger();
        $logger->log("=====================================");
        $logger->log("🔧 DIAGNOSE-TEST GESTARTET");
        $logger->log("=====================================");
        
        try {
            // Test 1: Verzeichnisse prüfen
            $dir_writable = is_writable(JBM_BACKUP_DIR);
            $logger->log("Backup-Dir beschreibbar: " . ($dir_writable ? 'JA' : 'NEIN'));
            
            // Test 2: Backup erstellen
            $backup = new \JBM\Backup();
            $logger->log("Backup-Klasse erstellt");
            
            $result = $backup->create('database', 'DIAGNOSE-TEST');
            $logger->log("Backup::create() Result: " . json_encode($result));
            
            // Test 3: Datei prüfen
            $backup_path = isset($result['backup_path']) ? $result['backup_path'] : '';
            $file_exists = $backup_path && file_exists($backup_path);
            $file_size = $file_exists ? filesize($backup_path) : 0;
            
            $logger->log("Datei existiert: " . ($file_exists ? 'JA' : 'NEIN'));
            $logger->log("Dateipfad: " . $backup_path);
            $logger->log("Dateigröße: " . $file_size);
            
            // Test 4: DB-Eintrag prüfen
            global $wpdb;
            $table_name = $wpdb->prefix . 'jbm_backups';
            $total_backups = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");
            
            $db_entry = null;
            $db_entry_exists = false;
            
            if (isset($result['backup_name'])) {
                $db_entry = $wpdb->get_row($wpdb->prepare(
                    "SELECT * FROM $table_name WHERE backup_name = %s",
                    $result['backup_name']
                ));
                $db_entry_exists = $db_entry ? true : false;
                
                $logger->log("DB-Eintrag vorhanden: " . ($db_entry_exists ? 'JA (ID: ' . $db_entry->id . ')' : 'NEIN'));
            }
            
            // Alle Backups holen
            $all_backups = $wpdb->get_results("SELECT id, backup_name, backup_size, backup_date FROM $table_name ORDER BY backup_date DESC LIMIT 5");
            $logger->log("Letzte 5 Backups in DB: " . count($all_backups));
            foreach ($all_backups as $bk) {
                $logger->log("  - ID {$bk->id}: {$bk->backup_name} ({$bk->backup_size} Bytes) - {$bk->backup_date}");
            }
            
            $logger->log("=====================================");
            
            wp_send_json_success([
                'backup_created' => isset($result['success']) && $result['success'],
                'backup_name' => $result['backup_name'] ?? 'N/A',
                'backup_path' => $backup_path,
                'file_exists' => $file_exists,
                'file_path' => $backup_path,
                'file_size' => $file_exists ? round($file_size / 1024 / 1024, 2) . ' MB' : '0 KB',
                'db_entry_exists' => $db_entry_exists,
                'db_id' => $db_entry ? $db_entry->id : 0,
                'total_backups' => $total_backups,
                'dir_writable' => $dir_writable,
                'recent_backups' => $all_backups,
                'error' => $result['message'] ?? null,
                'full_result' => $result
            ]);
            
        } catch (\Exception $e) {
            $logger->log("❌ DIAGNOSE-FEHLER: " . $e->getMessage(), 'error');
            $logger->log("Datei: " . $e->getFile() . ':' . $e->getLine(), 'error');
            
            wp_send_json_error([
                'message' => 'Kritischer Fehler: ' . $e->getMessage(),
                'file' => basename($e->getFile()) . ':' . $e->getLine()
            ]);
        }
    }
    
    /**
     * AJAX: Logs löschen
     */
    /**
     * AJAX: Nginx-Info ausblenden
     */
    public function ajax_hide_nginx_info() {
        check_ajax_referer('jbm_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Keine Berechtigung', 'jenva-backup-migration')]);
            return;
        }
        
        update_option('jbm_hide_nginx_info', true);
        
        wp_send_json_success([
            'message' => __('Nginx-Info wird nicht mehr angezeigt', 'jenva-backup-migration')
        ]);
    }
    
    public function ajax_clear_logs() {
        check_ajax_referer('jbm_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Keine Berechtigung', 'jenva-backup-migration')]);
            return;
        }
        
        try {
            $logger = new \JBM\Logger();
            $logger->clear_log();
            
            wp_send_json_success([
                'message' => __('Logs wurden erfolgreich gelöscht', 'jenva-backup-migration')
            ]);
        } catch (\Exception $e) {
            wp_send_json_error([
                'message' => __('Fehler beim Löschen der Logs', 'jenva-backup-migration') . ': ' . $e->getMessage()
            ]);
        }
    }
    
    /**
     * Löscht alte Backups basierend auf 3 Limits (manuell aufgerufen)
     */
    /**
     * Manuelle Bereinigung alter Backups
     * WICHTIG: Jede Kategorie wird STRIKT SEPARAT behandelt!
     * - Full: NUR 'full' (komplettes Backup mit DB + Dateien)
     * - Database: 'database', 'db-only'
     * - Andere: 'files-only', 'plugins', 'themes', 'uploads'
     * - Hochgeladene Backups (uploaded, merged_*, converted_*) werden NICHT automatisch bereinigt
     */
    private function cleanup_old_backups() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'jbm_backups';
        $deleted_count = 0;
        
        $settings = get_option('jbm_settings', [
            'max_full_backups' => 10,
            'max_database_backups' => 30,
            'max_other_backups' => 20
        ]);
        
        $max_full = intval($settings['max_full_backups'] ?? 10);
        $max_database = intval($settings['max_database_backups'] ?? 30);
        $max_other = intval($settings['max_other_backups'] ?? 20);
        
        $logger = new \JBM\Logger();
        $logger->log(sprintf(__('🧹 Manuelle Bereinigung: full=%d, db=%d, other=%d', 'jenva-backup-migration'), $max_full, $max_database, $max_other));
        
        // ============================================
        // KATEGORIE 1: Full Backups (NUR 'full' Typ!)
        // ============================================
        $full_backups = $wpdb->get_results(
            "SELECT * FROM $table_name 
            WHERE backup_type = 'full' 
            AND (protected = 0 OR protected IS NULL)
            ORDER BY backup_date ASC"
        );
        
        $logger->log("Full-Backups (nur 'full'): " . count($full_backups) . " gefunden, Limit: {$max_full}");
        
        if (count($full_backups) > $max_full) {
            $delete_count = count($full_backups) - $max_full;
            $to_delete = array_slice($full_backups, 0, $delete_count);
            
            foreach ($to_delete as $backup) {
                if (file_exists($backup->backup_path)) {
                    @unlink($backup->backup_path);
                }
                $wpdb->delete($table_name, ['id' => $backup->id], ['%d']);
                $deleted_count++;
                $logger->log(sprintf(__('Gelöscht (Full): %s', 'jenva-backup-migration'), $backup->backup_name));
            }
        }
        
        // ============================================
        // KATEGORIE 2: Database Backups
        // ============================================
        $db_backups = $wpdb->get_results(
            "SELECT * FROM $table_name 
            WHERE backup_type IN ('database', 'db-only') 
            AND (protected = 0 OR protected IS NULL)
            ORDER BY backup_date ASC"
        );
        
        $logger->log("DB-Backups: " . count($db_backups) . " gefunden, Limit: {$max_database}");
        
        if (count($db_backups) > $max_database) {
            $delete_count = count($db_backups) - $max_database;
            $to_delete = array_slice($db_backups, 0, $delete_count);
            
            foreach ($to_delete as $backup) {
                if (file_exists($backup->backup_path)) {
                    @unlink($backup->backup_path);
                }
                $wpdb->delete($table_name, ['id' => $backup->id], ['%d']);
                $deleted_count++;
                $logger->log(sprintf(__('Gelöscht (DB): %s', 'jenva-backup-migration'), $backup->backup_name));
            }
        }
        
        // ============================================
        // KATEGORIE 3: Andere Backups (files-only, plugins, themes, uploads)
        // WICHTIG: Hochgeladene Backups werden NICHT bereinigt!
        // ============================================
        $other_backups = $wpdb->get_results(
            "SELECT * FROM $table_name 
            WHERE backup_type IN ('files-only', 'plugins', 'themes', 'uploads', 'content') 
            AND (protected = 0 OR protected IS NULL)
            ORDER BY backup_date ASC"
        );
        
        $logger->log("Andere Backups (files/plugins/themes/uploads): " . count($other_backups) . " gefunden, Limit: {$max_other}");
        
        if (count($other_backups) > $max_other) {
            $delete_count = count($other_backups) - $max_other;
            $to_delete = array_slice($other_backups, 0, $delete_count);
            
            foreach ($to_delete as $backup) {
                if (file_exists($backup->backup_path)) {
                    @unlink($backup->backup_path);
                }
                $wpdb->delete($table_name, ['id' => $backup->id], ['%d']);
                $deleted_count++;
                $logger->log(sprintf(__('Gelöscht (Andere): %s', 'jenva-backup-migration'), $backup->backup_name));
            }
        }
        
        $logger->log(sprintf(__('🧹 Manuelle Bereinigung abgeschlossen: %d Backup(s) gelöscht', 'jenva-backup-migration'), $deleted_count));
        
        return [
            'success' => true,
            'message' => sprintf(__('%d alte Backup(s) gelöscht', 'jenva-backup-migration'), $deleted_count),
            'deleted_count' => $deleted_count
        ];
    }
    
    /**
     * AJAX Handler für Bricks-Diagnose
     * BRICKS-KOMPATIBILITÄT: Führt umfassende Diagnose der Bricks-Daten durch
     */
    public function ajax_diagnose_bricks() {
        check_ajax_referer('jbm_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Keine Berechtigung', 'jenva-backup-migration')]);
        }
        
        global $wpdb;
        $logger = new \JBM\Logger();
        $restore = new \JBM\Restore();
        
        $logger->log("=====================================");
        $logger->log("🧱 BRICKS-DIAGNOSE GESTARTET");
        $logger->log("=====================================");
        
        try {
            // Verwende Reflection, um private Methoden aufzurufen
            $reflection = new \ReflectionClass($restore);
            
            // Prüfe ob Bricks-Daten vorhanden sind
            $check_method = $reflection->getMethod('check_for_bricks_data');
            $check_method->setAccessible(true);
            $has_bricks_data = $check_method->invoke($restore);
            
            if (!$has_bricks_data) {
                $logger->log("⚠️ Keine Bricks-Daten gefunden!");
                wp_send_json_success([
                    'message' => __('Keine Bricks-Daten gefunden', 'jenva-backup-migration'),
                    'has_bricks_data' => false
                ]);
                return;
            }
            
            // Führe umfassende Diagnose durch
            $diagnose_method = $reflection->getMethod('diagnose_bricks_data');
            $diagnose_method->setAccessible(true);
            $diagnose_method->invoke($restore, []);
            
            // Führe Korrektur durch
            $logger->log("🔧 Führe Bricks-Optionen-Korrektur durch...");
            $fix_method = $reflection->getMethod('fix_all_bricks_options');
            $fix_method->setAccessible(true);
            $fix_method->invoke($restore);
            
            // Verifiziere postmeta-Daten
            $logger->log("🔍 Verifiziere Bricks postmeta-Daten...");
            $verify_method = $reflection->getMethod('verify_and_fix_bricks_postmeta');
            $verify_method->setAccessible(true);
            $verify_method->invoke($restore);
            
            $logger->log("=====================================");
            $logger->log("✅ BRICKS-DIAGNOSE ABGESCHLOSSEN");
            $logger->log("=====================================");
            
            wp_send_json_success([
                'message' => __('Bricks-Diagnose abgeschlossen. Bitte prüfen Sie die Logs.', 'jenva-backup-migration'),
                'has_bricks_data' => true
            ]);
            
        } catch (\Exception $e) {
            $logger->log("❌ Bricks-Diagnose Fehler: " . $e->getMessage(), 'error');
            wp_send_json_error([
                'message' => $e->getMessage()
            ]);
        }
    }
}

// Plugin initialisieren
function jbm_init() {
    return Jenva_Backup_Migration::instance();
}

add_action('plugins_loaded', 'jbm_init');

